"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getColor = exports.byDataColorPaletteMap = void 0;
var _lodash = require("lodash");
var _public = require("@kbn/charts-plugin/public");
var _types = require("../../../common/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const isTreemapOrMosaicChart = shape => [_types.ChartTypes.MOSAIC, _types.ChartTypes.TREEMAP].includes(shape);
const byDataColorPaletteMap = (paletteDefinition, {
  params
}, colorIndexMap) => {
  const colorCache = new Map();
  return {
    getColor: item => {
      var _colorIndexMap$get;
      const key = String(item);
      let color = colorCache.get(key);
      if (color) return color;
      const colorIndex = (_colorIndexMap$get = colorIndexMap.get(key)) !== null && _colorIndexMap$get !== void 0 ? _colorIndexMap$get : -1;
      color = paletteDefinition.getCategoricalColor([{
        name: key,
        totalSeriesAtDepth: colorIndexMap.size,
        rankAtDepth: colorIndex
      }], {
        behindText: false
      }, params) || undefined;
      colorCache.set(key, color);
      return color;
    }
  };
};
exports.byDataColorPaletteMap = byDataColorPaletteMap;
const getDistinctColor = (categoricalKey, isSplitChart, overwriteColors = {}, visParams, paletteService, syncColors, {
  parentSeries,
  allSeries
}, formattedCategoricalKey) => {
  var _visParams$palette$pa, _visParams$palette;
  // TODO move away from Record to a Map to avoid issues with reserved JS keywords
  if (Object.hasOwn(overwriteColors, categoricalKey)) {
    return overwriteColors[categoricalKey];
  }
  // this is for supporting old visualizations (created by vislib plugin)
  // it seems that there for some aggs, the uiState saved from vislib is
  // different from how es-charts handles it
  if (Object.hasOwn(overwriteColors, formattedCategoricalKey)) {
    return overwriteColors[formattedCategoricalKey];
  }
  const index = allSeries.findIndex(d => (0, _lodash.isEqual)(d, categoricalKey));
  const isSplitParentLayer = isSplitChart && parentSeries.includes(categoricalKey);
  return paletteService === null || paletteService === void 0 ? void 0 : paletteService.get(visParams.palette.name).getCategoricalColor([{
    name: categoricalKey,
    rankAtDepth: isSplitParentLayer ? parentSeries.findIndex(d => d === categoricalKey) : index > -1 ? index : 0,
    totalSeriesAtDepth: isSplitParentLayer ? parentSeries.length : allSeries.length || 1
  }], {
    maxDepth: 1,
    totalSeries: allSeries.length || 1,
    behindText: visParams.labels.show,
    syncColors
  }, (_visParams$palette$pa = (_visParams$palette = visParams.palette) === null || _visParams$palette === void 0 ? void 0 : _visParams$palette.params) !== null && _visParams$palette$pa !== void 0 ? _visParams$palette$pa : {
    colors: []
  });
};

/**
 * This interface is introduced to simplify the used logic on testing an ArrayNode outside elastic-charts.
 * The SimplifiedArrayNode structure resembles the hierarchical configuration of an ArrayNode,
 * by presenting only the necessary fields used by the functions in this file.
 * The main difference is in the parent node, that to simplify this infinite tree structure we configured it as optional
 * so that in test I don't need to add a type assertion on an undefined parent as in elastic-charts.
 * The children are slight different in implementation and they accept `unknown` as key
 * due to the situation described in https://github.com/elastic/kibana/issues/153437
 */

/**
 * This method returns the path of each hierarchical layer encountered from the given node
 * (a node of a hierarchical tree, currently a partition tree) up to the root of the hierarchy tree.
 * The resulting array only shows, for each parent, the name of the node, its child index within the parent branch
 * (called rankInDepth) and the total number of children of the parent.
 */
const createSeriesLayers = (arrayNode, parentSeries, isSplitChart, colorIndexMap) => {
  const seriesLayers = [];
  let tempParent = arrayNode;
  while (tempParent.parent && tempParent.depth > 0) {
    var _colorIndexMap$get2;
    const nodeKey = tempParent.parent.children[tempParent.sortIndex][0];
    const seriesName = String(nodeKey);

    /**
     * FIXME this is a bad implementation: The `parentSeries` is an array of both `string` and `RangeKey` even if its type
     * is marked as `string[]` in `DistinctSeries`. Here instead we are checking if a stringified `RangeKey` is included into this array that
     * is conceptually wrong.
     * see https://github.com/elastic/kibana/issues/153437
     */
    const isSplitParentLayer = isSplitChart && parentSeries.includes(seriesName);
    const colorIndex = (_colorIndexMap$get2 = colorIndexMap.get(seriesName)) !== null && _colorIndexMap$get2 !== void 0 ? _colorIndexMap$get2 : tempParent.sortIndex;
    seriesLayers.unshift({
      name: seriesName,
      rankAtDepth: isSplitParentLayer ?
      // FIXME as described above this will not work correctly if the `nodeKey` is a `RangeKey`
      parentSeries.findIndex(name => name === seriesName) : colorIndex,
      totalSeriesAtDepth: isSplitParentLayer ? parentSeries.length : tempParent.parent.children.length
    });
    tempParent = tempParent.parent;
  }
  return seriesLayers;
};
const overrideColors = (seriesLayers, overwriteColors, name) => {
  let overwriteColor;
  if (Object.hasOwn(overwriteColors, name)) {
    overwriteColor = overwriteColors[name];
  }
  seriesLayers.forEach(layer => {
    if (Object.keys(overwriteColors).includes(layer.name)) {
      overwriteColor = overwriteColors[layer.name];
    }
  });
  return overwriteColor;
};
const getColor = (chartType, categoricalKey, arrayNode, layerIndex, isSplitChart, overwriteColors = {}, distinctSeries, {
  columnsLength,
  rowsLength
}, visParams, paletteService, byDataPalette, syncColors, isDarkMode, formatter, column, colorIndexMap) => {
  var _formatter$deserializ, _visParams$palette$pa2, _visParams$palette2;
  // Mind the difference here: the contrast computation for the text ignores the alpha/opacity
  // therefore change it for dark mode
  const defaultColor = isDarkMode ? 'rgba(0,0,0,0)' : 'rgba(255,255,255,0)';
  const name = column.format ? (_formatter$deserializ = formatter.deserialize(column.format).convert(categoricalKey)) !== null && _formatter$deserializ !== void 0 ? _formatter$deserializ : '' : '';
  if (visParams.distinctColors) {
    return getDistinctColor(categoricalKey, isSplitChart, overwriteColors, visParams, paletteService, syncColors, distinctSeries, name) || defaultColor;
  }
  const seriesLayers = createSeriesLayers(arrayNode, distinctSeries.parentSeries, isSplitChart, colorIndexMap);
  const overriddenColor = overrideColors(seriesLayers, overwriteColors, name);
  if (overriddenColor) {
    // this is necessary for supporting some old visualizations that defined their own colors (created by vislib plugin)
    return (0, _public.lightenColor)(overriddenColor, seriesLayers.length, columnsLength);
  }
  if (chartType === _types.ChartTypes.MOSAIC && byDataPalette && seriesLayers[1]) {
    return byDataPalette.getColor(seriesLayers[1].name) || defaultColor;
  }
  if (chartType === _types.ChartTypes.MOSAIC && layerIndex < columnsLength - 1) {
    return defaultColor;
  }

  //  Mosaic - use the second layer for color
  if (chartType === _types.ChartTypes.MOSAIC && seriesLayers.length > 1) {
    seriesLayers.shift();
  }
  const outputColor = paletteService === null || paletteService === void 0 ? void 0 : paletteService.get(visParams.palette.name).getCategoricalColor(seriesLayers, {
    behindText: visParams.labels.show || isTreemapOrMosaicChart(chartType),
    maxDepth: columnsLength,
    totalSeries: rowsLength,
    syncColors
  }, (_visParams$palette$pa2 = (_visParams$palette2 = visParams.palette) === null || _visParams$palette2 === void 0 ? void 0 : _visParams$palette2.params) !== null && _visParams$palette$pa2 !== void 0 ? _visParams$palette$pa2 : {
    colors: []
  });
  return outputColor || defaultColor;
};
exports.getColor = getColor;