"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DiscoverSearchSessionManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _public = require("@kbn/kibana-utils-plugin/public");
var _queryString = require("query-string");
var _constants = require("../../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Helps with state management of search session and {@link SEARCH_SESSION_ID_QUERY_PARAM} in the URL
 */
class DiscoverSearchSessionManager {
  constructor(deps) {
    (0, _defineProperty2.default)(this, "deps", void 0);
    (0, _defineProperty2.default)(this, "getSearchSessionIdFromURL", () => (0, _public.getQueryParams)(this.deps.history.location)[_constants.SEARCH_SESSION_ID_QUERY_PARAM]);
    this.deps = deps;
  }

  /**
   * Notifies about `searchSessionId` changes in the URL,
   * skips if `searchSessionId` matches current search session id
   */
  getNewSearchSessionIdFromURL$() {
    return (0, _public.createQueryParamObservable)(this.deps.history, _constants.SEARCH_SESSION_ID_QUERY_PARAM).pipe((0, _rxjs.filter)(searchSessionId => {
      if (!searchSessionId) return true;
      return !this.deps.session.isCurrentSession(searchSessionId);
    }));
  }

  /**
   * Get next session id by either starting or restoring a session.
   * When navigating away from the restored session {@link SEARCH_SESSION_ID_QUERY_PARAM} is removed from the URL using history.replace
   */
  getNextSearchSessionId() {
    var _searchSessionIdFromU;
    let searchSessionIdFromURL = this.getSearchSessionIdFromURL();
    if (searchSessionIdFromURL) {
      if (this.deps.session.isRestore() && this.deps.session.isCurrentSession(searchSessionIdFromURL)) {
        // navigating away from a restored session
        this.removeSearchSessionIdFromURL({
          replace: true
        });
        searchSessionIdFromURL = undefined;
      } else {
        this.deps.session.restore(searchSessionIdFromURL);
      }
    }
    return {
      searchSessionId: (_searchSessionIdFromU = searchSessionIdFromURL) !== null && _searchSessionIdFromU !== void 0 ? _searchSessionIdFromU : this.deps.session.start(),
      isSearchSessionRestored: Boolean(searchSessionIdFromURL)
    };
  }

  /**
   * Pushes the provided search session ID to the URL
   * @param searchSessionId - the search session ID to push to the URL
   */
  pushSearchSessionIdToURL(searchSessionId, {
    replace = true
  } = {
    replace: true
  }) {
    const oldLocation = this.deps.history.location;
    const query = (0, _public.getQueryParams)(oldLocation);
    query[_constants.SEARCH_SESSION_ID_QUERY_PARAM] = searchSessionId;
    const newSearch = (0, _queryString.stringify)(_public.url.encodeQuery(query), {
      sort: false,
      encode: false
    });
    const newLocation = {
      ...oldLocation,
      search: newSearch
    };
    if (replace) {
      this.deps.history.replace(newLocation);
    } else {
      this.deps.history.push(newLocation);
    }
  }

  /**
   * Removes Discovers {@link SEARCH_SESSION_ID_QUERY_PARAM} from the URL
   * @param replace - methods to change the URL
   */
  removeSearchSessionIdFromURL({
    replace = true
  } = {
    replace: true
  }) {
    if (this.hasSearchSessionIdInURL()) {
      (0, _public.removeQueryParam)(this.deps.history, _constants.SEARCH_SESSION_ID_QUERY_PARAM, replace);
    }
  }

  /**
   * If there is a {@link SEARCH_SESSION_ID_QUERY_PARAM} currently in the URL
   */
  hasSearchSessionIdInURL() {
    return !!this.getSearchSessionIdFromURL();
  }
}
exports.DiscoverSearchSessionManager = DiscoverSearchSessionManager;