"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.STATUS_LEVEL_UI_ATTRS = void 0;
exports.loadStatus = loadStatus;
var _numeral = _interopRequireDefault(require("@elastic/numeral"));
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns an object of any keys that should be included for metrics.
 */
function formatMetrics({
  metrics
}) {
  var _metrics$process$even, _metrics$process$even2, _metrics$process$even3;
  if (!metrics) {
    return [];
  }
  return [{
    name: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.heapUsedHeader', {
      defaultMessage: 'Heap used out of {heapTotal}',
      values: {
        heapTotal: (0, _numeral.default)(metrics.process.memory.heap.size_limit).format('0.00 b')
      }
    }),
    value: metrics.process.memory.heap.used_in_bytes,
    type: 'byte'
  }, {
    name: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.requestsPerSecHeader', {
      defaultMessage: 'Requests per second'
    }),
    value: metrics.requests.total * 1000 / metrics.collection_interval_in_millis,
    type: 'float'
  }, {
    name: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.utilizationHeader', {
      defaultMessage: 'Utilization (active: {active} / idle: {idle})',
      values: {
        active: (0, _numeral.default)(metrics.process.event_loop_utilization.active).format('0.00'),
        idle: (0, _numeral.default)(metrics.process.event_loop_utilization.idle).format('0.00')
      }
    }),
    value: metrics.process.event_loop_utilization.utilization,
    type: 'float'
  }, {
    name: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.loadHeader', {
      defaultMessage: 'Load'
    }),
    value: [metrics.os.load['1m'], metrics.os.load['5m'], metrics.os.load['15m']],
    type: 'float',
    meta: {
      description: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.load.metaHeader', {
        defaultMessage: 'Load interval'
      }),
      title: Object.keys(metrics.os.load).join('; ')
    }
  }, {
    name: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.processDelayHeader', {
      defaultMessage: 'Delay'
    }),
    value: metrics.process.event_loop_delay,
    type: 'time',
    meta: {
      description: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.processDelayDetailsHeader', {
        defaultMessage: 'Percentiles'
      }),
      title: '',
      value: [(_metrics$process$even = metrics.process.event_loop_delay_histogram) === null || _metrics$process$even === void 0 ? void 0 : _metrics$process$even.percentiles['50'], (_metrics$process$even2 = metrics.process.event_loop_delay_histogram) === null || _metrics$process$even2 === void 0 ? void 0 : _metrics$process$even2.percentiles['95'], (_metrics$process$even3 = metrics.process.event_loop_delay_histogram) === null || _metrics$process$even3 === void 0 ? void 0 : _metrics$process$even3.percentiles['99']],
      type: 'time'
    }
  }, {
    name: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.resTimeAvgHeader', {
      defaultMessage: 'Response time avg'
    }),
    value: metrics.response_times.avg_in_millis,
    type: 'time',
    meta: {
      description: _i18n.i18n.translate('core.statusPage.metricsTiles.columns.resTimeMaxHeader', {
        defaultMessage: 'Response time max'
      }),
      title: '',
      value: [metrics.response_times.max_in_millis],
      type: 'time'
    }
  }];
}

/**
 * Reformat the backend data to make the frontend views simpler.
 */
function formatStatus(id, status) {
  const {
    title,
    uiColor
  } = STATUS_LEVEL_UI_ATTRS[status.level];
  return {
    id,
    original: status,
    state: {
      id: status.level,
      message: status.summary,
      title,
      uiColor
    }
  };
}
/**
 * Maps status to health color for frontend views
 */
const STATUS_LEVEL_UI_ATTRS = exports.STATUS_LEVEL_UI_ATTRS = {
  critical: {
    title: _i18n.i18n.translate('core.status.redTitle', {
      defaultMessage: 'Red'
    }),
    uiColor: 'danger'
  },
  unavailable: {
    title: _i18n.i18n.translate('core.status.redTitle', {
      defaultMessage: 'Red'
    }),
    uiColor: 'danger'
  },
  degraded: {
    title: _i18n.i18n.translate('core.status.yellowTitle', {
      defaultMessage: 'Yellow'
    }),
    uiColor: 'warning'
  },
  available: {
    title: _i18n.i18n.translate('core.status.greenTitle', {
      defaultMessage: 'Green'
    }),
    uiColor: 'success'
  }
};

/**
 * Get the status from the server API and format it for display.
 */
async function loadStatus({
  http,
  notifications
}) {
  let response;
  try {
    response = await http.get('/api/status');
  } catch (e) {
    var _e$response, _e$body;
    // API returns a 503 response if not all services are available.
    // In this case, we want to treat this as a successful API call, so that we can
    // display Kibana's status correctly.
    // 503 responses can happen for other reasons (such as proxies), so we make an educated
    // guess here to determine if the response payload looks like an appropriate `StatusResponse`.
    const ignoreError = ((_e$response = e.response) === null || _e$response === void 0 ? void 0 : _e$response.status) === 503 && typeof ((_e$body = e.body) === null || _e$body === void 0 ? void 0 : _e$body.name) === 'string';
    if (ignoreError) {
      response = e.body;
    } else {
      var _e$response$status, _e$response2;
      if (((_e$response$status = (_e$response2 = e.response) === null || _e$response2 === void 0 ? void 0 : _e$response2.status) !== null && _e$response$status !== void 0 ? _e$response$status : 0) >= 400) {
        var _e$response3;
        notifications.toasts.addDanger(_i18n.i18n.translate('core.statusPage.loadStatus.serverStatusCodeErrorMessage', {
          defaultMessage: 'Failed to request server status with status code {responseStatus}',
          values: {
            responseStatus: (_e$response3 = e.response) === null || _e$response3 === void 0 ? void 0 : _e$response3.status
          }
        }));
      } else {
        notifications.toasts.addDanger(_i18n.i18n.translate('core.statusPage.loadStatus.serverIsDownErrorMessage', {
          defaultMessage: 'Failed to request server status. Perhaps your server is down?'
        }));
      }
      throw e;
    }
  }
  return {
    name: response.name,
    version: response.version,
    coreStatus: Object.entries(response.status.core).map(([serviceName, status]) => formatStatus(serviceName, status)),
    pluginStatus: Object.entries(response.status.plugins).map(([pluginName, status]) => formatStatus(pluginName, status)),
    serverState: formatStatus('overall', response.status.overall).state,
    metrics: formatMetrics(response)
  };
}