"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGetAllRoute = registerGetAllRoute;
exports.registerGetOneRoute = registerGetOneRoute;
var _configSchema = require("@kbn/config-schema");
var _lib = require("../../../../common/lib");
var _get_managed_templates = require("../../../lib/get_managed_templates");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerGetAllRoute({
  router,
  config,
  lib: {
    handleEsError
  }
}) {
  router.get({
    path: (0, _.addBasePath)('/index_templates'),
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: false
  }, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      const cloudManagedTemplatePrefix = await (0, _get_managed_templates.getCloudManagedTemplatePrefix)(client);
      const {
        index_templates: templatesEs
      } = await client.asCurrentUser.indices.getIndexTemplate();
      const templates = (0, _lib.deserializeTemplateList)(
      // @ts-expect-error TemplateSerialized.index_patterns not compatible with IndicesIndexTemplate.index_patterns
      templatesEs, cloudManagedTemplatePrefix);
      if (config.isLegacyTemplatesEnabled === false) {
        // If isLegacyTemplatesEnabled=false, we do not want to fetch legacy templates and return an empty array;
        // we retain the same response format to limit changes required on the client
        return response.ok({
          body: {
            templates,
            legacyTemplates: []
          }
        });
      }
      const legacyTemplatesEs = await client.asCurrentUser.indices.getTemplate();
      const legacyTemplates = (0, _lib.deserializeLegacyTemplateList)(legacyTemplatesEs, cloudManagedTemplatePrefix);
      const body = {
        templates,
        legacyTemplates
      };
      return response.ok({
        body
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
}
const paramsSchema = _configSchema.schema.object({
  name: _configSchema.schema.string()
});

// Require the template format version (V1 or V2) to be provided as Query param
const querySchema = _configSchema.schema.object({
  legacy: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('true'), _configSchema.schema.literal('false')]))
});
function registerGetOneRoute({
  router,
  config,
  lib: {
    handleEsError
  }
}) {
  router.get({
    path: (0, _.addBasePath)('/index_templates/{name}'),
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {
      params: paramsSchema,
      query: querySchema
    }
  }, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      name
    } = request.params;
    // We don't expect the `legacy` query to be used when legacy templates are disabled, however, we add the `enableLegacyTemplates` check as a safeguard
    const isLegacy = config.isLegacyTemplatesEnabled !== false && request.query.legacy === 'true';
    try {
      var _persistent$cluster$l, _persistent$cluster, _persistent$cluster$l2, _defaults$cluster, _defaults$cluster$log;
      const cloudManagedTemplatePrefix = await (0, _get_managed_templates.getCloudManagedTemplatePrefix)(client);
      const {
        persistent,
        defaults
      } = await client.asInternalUser.cluster.getSettings({
        include_defaults: true
      });
      const isLogsdbEnabled = ((_persistent$cluster$l = persistent === null || persistent === void 0 ? void 0 : (_persistent$cluster = persistent.cluster) === null || _persistent$cluster === void 0 ? void 0 : (_persistent$cluster$l2 = _persistent$cluster.logsdb) === null || _persistent$cluster$l2 === void 0 ? void 0 : _persistent$cluster$l2.enabled) !== null && _persistent$cluster$l !== void 0 ? _persistent$cluster$l : defaults === null || defaults === void 0 ? void 0 : (_defaults$cluster = defaults.cluster) === null || _defaults$cluster === void 0 ? void 0 : (_defaults$cluster$log = _defaults$cluster.logsdb) === null || _defaults$cluster$log === void 0 ? void 0 : _defaults$cluster$log.enabled) === 'true';
      if (isLegacy) {
        const indexTemplateByName = await client.asCurrentUser.indices.getTemplate({
          name
        });
        if (indexTemplateByName[name]) {
          return response.ok({
            body: (0, _lib.deserializeLegacyTemplate)({
              ...indexTemplateByName[name],
              name
            }, cloudManagedTemplatePrefix, isLogsdbEnabled)
          });
        }
      } else {
        const {
          index_templates: indexTemplates
        } = await client.asCurrentUser.indices.getIndexTemplate({
          name
        });
        if (indexTemplates.length > 0) {
          return response.ok({
            body: (0, _lib.deserializeTemplate)(
            // @ts-expect-error TemplateSerialized.index_patterns not compatible with IndicesIndexTemplate.index_patterns
            {
              ...indexTemplates[0].index_template,
              name
            }, cloudManagedTemplatePrefix, isLogsdbEnabled)
          });
        }
      }
      return response.notFound();
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
}