"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validate = void 0;
var _is = require("../../../ast/is");
var _errors = require("../../../definitions/utils/errors");
var _validation = require("../../../definitions/utils/validation");
var _util = require("./util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const validate = (command, ast, context, callbacks) => {
  const messages = [];
  const source = command.args[0];
  const cluster = source.prefix;
  const index = source.index;
  const policies = (context === null || context === void 0 ? void 0 : context.policies) || new Map();
  if (index && !policies.has(index.valueUnquoted)) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unknownPolicy',
      values: {
        name: index.valueUnquoted
      },
      locations: index.location
    }));
  }
  if (cluster) {
    const acceptedModes = new Set(_util.ENRICH_MODES.map(({
      name
    }) => '_' + name.toLowerCase()));
    const isValidMode = acceptedModes.has(cluster.valueUnquoted.toLowerCase());
    if (!isValidMode) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unsupportedMode',
        values: {
          command: 'ENRICH',
          value: cluster.valueUnquoted,
          expected: [...acceptedModes].join(', ')
        },
        locations: cluster.location
      }));
    }
  }
  const policy = index && policies.get(index.valueUnquoted);
  const withOption = command.args.find(arg => (0, _is.isOptionNode)(arg) && arg.name === 'with');
  if (withOption && policy) {
    withOption.args.forEach(arg => {
      if ((0, _is.isAssignment)(arg) && Array.isArray(arg.args[1]) && (0, _is.isColumn)(arg.args[1][0])) {
        const column = arg.args[1][0];
        if (!policy.enrichFields.includes(column.parts.join('.'))) {
          messages.push(_errors.errors.unknownColumn(column));
        }
      }
    });
  }
  messages.push(...(0, _validation.validateCommandArguments)({
    ...command,
    // exclude WITH from generic validation since it shouldn't be compared against the generic column list
    args: command.args.filter(arg => arg !== withOption)
  }, ast, context, callbacks));
  return messages;
};
exports.validate = validate;