"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerPackagesBulkOperationTask = registerPackagesBulkOperationTask;
var _services = require("../../services");
var _run_bulk_upgrade = require("./run_bulk_upgrade");
var _run_bulk_uninstall = require("./run_bulk_uninstall");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerPackagesBulkOperationTask(taskManager) {
  taskManager.registerTaskDefinitions({
    [_utils.TASK_TYPE]: {
      title: _utils.TASK_TITLE,
      timeout: _utils.TASK_TIMEOUT,
      createTaskRunner: ({
        taskInstance
      }) => {
        const abortController = new AbortController();
        const logger = _services.appContextService.getLogger();
        return {
          run: async () => {
            logger.debug(`Starting bulk package operations: ${taskInstance.params.type}`);
            if (taskInstance.state.isDone) {
              return;
            }
            const taskParams = taskInstance.params;
            try {
              let results;
              if (taskParams.type === 'bulk_uninstall') {
                results = await (0, _run_bulk_uninstall._runBulkUninstallTask)({
                  abortController,
                  logger,
                  taskParams: taskParams
                });
              } else if (taskParams.type === 'bulk_upgrade') {
                results = await (0, _run_bulk_upgrade._runBulkUpgradeTask)({
                  abortController,
                  logger,
                  taskParams: taskParams
                });
              }
              const state = {
                isDone: true,
                results
              };
              return {
                runAt: new Date(Date.now() + 60 * 60 * 1000),
                state
              };
            } catch (error) {
              logger.error(`Packages bulk operation: ${taskParams.type} failed`, {
                error
              });
              return {
                runAt: new Date(Date.now() + 60 * 60 * 1000),
                state: {
                  isDone: true,
                  error: (0, _utils.formatError)(error)
                }
              };
            }
          },
          cancel: async () => {
            logger.debug(`Bulk package operations timed out: ${taskInstance.params.type}`);
            abortController.abort('task timed out');
          }
        };
      }
    }
  });
}