"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.TYPE = exports.AutomaticAgentUpgradeTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _task = require("@kbn/task-manager-plugin/server/task");
var _elasticsearch = require("@elastic/elasticsearch");
var _gt = _interopRequireDefault(require("semver/functions/gt"));
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../common/constants");
var _services = require("../services");
var _agents = require("../services/agents");
var _constants2 = require("../constants");
var _services2 = require("../../common/services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'fleet:automatic-agent-upgrade-task';
const VERSION = exports.VERSION = '1.0.2';
const TITLE = 'Fleet Automatic agent upgrades';
const SCOPE = ['fleet'];
const DEFAULT_INTERVAL = '30m';
const TIMEOUT = '10m';
const AGENT_POLICIES_BATCHSIZE = 500;
const AGENTS_BATCHSIZE = 10000;
const MIN_AGENTS_FOR_ROLLOUT = 10;
const MIN_UPGRADE_DURATION_SECONDS = 600;
class AutomaticAgentUpgradeTask {
  constructor(setupContract) {
    var _config$taskInterval, _config$retryDelays;
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "taskInterval", void 0);
    (0, _defineProperty2.default)(this, "retryDelays", void 0);
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager
    }) => {
      if (!taskManager) {
        this.logger.error('[AutomaticAgentUpgradeTask] Missing required service during start');
        return;
      }
      this.wasStarted = true;
      this.logger.info(`[AutomaticAgentUpgradeTask] Started with interval of [${this.taskInterval}]`);
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: TYPE,
          scope: SCOPE,
          schedule: {
            interval: this.taskInterval
          },
          state: {},
          params: {
            version: VERSION
          }
        });
      } catch (e) {
        this.logger.error(`Error scheduling task AutomaticAgentUpgradeTask, error: ${e.message}`, e);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core) => {
      if (!_services.appContextService.getExperimentalFeatures().enableAutomaticAgentUpgrades) {
        this.logger.debug('[AutomaticAgentUpgradeTask] Aborting runTask: automatic upgrades feature is disabled');
        return;
      }
      if (!_services.licenseService.isEnterprise()) {
        this.logger.debug('[AutomaticAgentUpgradeTask] Aborting runTask: automatic upgrades feature requires at least Enterprise license');
        return;
      }
      if (!this.wasStarted) {
        this.logger.debug('[AutomaticAgentUpgradeTask] Aborting runTask(): task not started yet');
        return;
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        this.logger.debug(`[AutomaticAgentUpgradeTask] Outdated task version: Got [${taskInstance.id}] from task instance. Current version is [${this.taskId}]`);
        return (0, _task.getDeleteTaskRunResult)();
      }
      this.logger.info('[AutomaticAgentUpgradeTask] runTask() started');
      const [coreStart] = await core.getStartServices();
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = new _server.SavedObjectsClient(coreStart.savedObjects.createInternalRepository());
      try {
        await this.checkAgentPoliciesForAutomaticUpgrades(esClient, soClient);
        this.endRun('success');
      } catch (err) {
        if (err instanceof _elasticsearch.errors.RequestAbortedError) {
          this.logger.warn(`[AutomaticAgentUpgradeTask] Request aborted due to timeout: ${err}`);
          this.endRun();
          return;
        }
        this.logger.error(`[AutomaticAgentUpgradeTask] Error: ${err}`);
        this.endRun('error');
      }
    });
    const {
      core: _core,
      taskManager: _taskManager,
      logFactory,
      config
    } = setupContract;
    this.logger = logFactory.get(this.taskId);
    this.taskInterval = (_config$taskInterval = config.taskInterval) !== null && _config$taskInterval !== void 0 ? _config$taskInterval : DEFAULT_INTERVAL;
    this.retryDelays = (_config$retryDelays = config.retryDelays) !== null && _config$retryDelays !== void 0 ? _config$retryDelays : _constants.AUTO_UPGRADE_DEFAULT_RETRIES;
    _taskManager.registerTaskDefinitions({
      [TYPE]: {
        title: TITLE,
        timeout: TIMEOUT,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask(taskInstance, _core);
            },
            cancel: async () => {
              this.abortController.abort('Task timed out');
            }
          };
        }
      }
    });
  }
  get taskId() {
    return `${TYPE}:${VERSION}`;
  }
  endRun(msg = '') {
    this.logger.info(`[AutomaticAgentUpgradeTask] runTask() ended${msg ? ': ' + msg : ''}`);
  }
  throwIfAborted() {
    if (this.abortController.signal.aborted) {
      throw new Error('Task was aborted');
    }
  }
  async checkAgentPoliciesForAutomaticUpgrades(esClient, soClient) {
    // Fetch custom agent policies with set required_versions in batches.
    const agentPolicyFetcher = await _services.agentPolicyService.fetchAllAgentPolicies(soClient, {
      kuery: `${_constants2.AGENT_POLICY_SAVED_OBJECT_TYPE}.is_managed:false AND ${_constants2.AGENT_POLICY_SAVED_OBJECT_TYPE}.required_versions:*`,
      perPage: AGENT_POLICIES_BATCHSIZE,
      fields: ['id', 'required_versions'],
      spaceId: '*'
    });
    for await (const agentPolicyPageResults of agentPolicyFetcher) {
      this.logger.debug(`[AutomaticAgentUpgradeTask] Found ${agentPolicyPageResults.length} agent policies with required_versions`);
      if (!agentPolicyPageResults.length) {
        this.endRun('Found no agent policies to process');
        return;
      }
      for (const agentPolicy of agentPolicyPageResults) {
        this.throwIfAborted();
        await this.checkAgentPolicyForAutomaticUpgrades(esClient, soClient, agentPolicy);
      }
    }
  }
  async checkAgentPolicyForAutomaticUpgrades(esClient, soClient, agentPolicy) {
    this.logger.debug(`[AutomaticAgentUpgradeTask] Processing agent policy ${agentPolicy.id} with required_versions ${JSON.stringify(agentPolicy.required_versions)}`);

    // Get total number of active agents.
    // This is used to calculate how many agents should be selected for upgrade based on the target percentage.
    const totalActiveAgents = await this.getAgentCount(esClient, soClient, `policy_id:${agentPolicy.id} AND ${_services2.AgentStatusKueryHelper.buildKueryForActiveAgents()}`);
    if (totalActiveAgents === 0) {
      this.logger.debug(`[AutomaticAgentUpgradeTask] Agent policy ${agentPolicy.id} has no active agents`);
      return;
    }
    // Before processing each required version, we need to get the count of agents for each version so we know if we should round some up or down to make sure we arent overshooting the total number of agents.
    const versionAndCounts = await this.getVersionAndCounts(agentPolicy, totalActiveAgents, esClient, soClient);
    for (const requiredVersion of (_agentPolicy$required = agentPolicy.required_versions) !== null && _agentPolicy$required !== void 0 ? _agentPolicy$required : []) {
      var _agentPolicy$required;
      await this.processRequiredVersion(esClient, soClient, agentPolicy, requiredVersion, versionAndCounts);
    }
  }
  async getVersionAndCounts(agentPolicy, totalActiveAgents, esClient, soClient) {
    let versionAndCounts = [];
    for (const requiredVersion of (_agentPolicy$required2 = agentPolicy.required_versions) !== null && _agentPolicy$required2 !== void 0 ? _agentPolicy$required2 : []) {
      var _agentPolicy$required2;
      let numberOfAgentsForUpgrade = Math.round(totalActiveAgents * requiredVersion.percentage / 100);

      // Subtract the total number of agents already or on or updating to target version.
      const updatingToKuery = `(upgrade_details.target_version:${requiredVersion.version} AND NOT upgrade_details.state:UPG_FAILED)`;
      const totalOnOrUpdatingToTargetVersionAgents = await this.getAgentCount(esClient, soClient, `((policy_id:${agentPolicy.id} AND agent.version:${requiredVersion.version}) OR ${updatingToKuery}) AND ${_services2.AgentStatusKueryHelper.buildKueryForActiveAgents()}`);
      numberOfAgentsForUpgrade -= totalOnOrUpdatingToTargetVersionAgents;
      versionAndCounts.push({
        version: requiredVersion.version,
        count: numberOfAgentsForUpgrade,
        targetPercentage: requiredVersion.percentage,
        alreadyUpgrading: totalOnOrUpdatingToTargetVersionAgents
      });
    }
    // Then we need to make adjustments based on the total to make sure we arent over or undershooting the total number of agents
    versionAndCounts = await this.adjustAgentCounts(versionAndCounts, totalActiveAgents);
    return versionAndCounts;
  }
  async adjustAgentCounts(versionAndCounts, totalActiveAgents) {
    //  Calculate what we actually have vs what we need to have.
    //  First we need to get the total actual percentage if we actually added the new agents and considering the existing ones
    const totalActualPercentage = (versionAndCounts.reduce((acc, item) => acc + item.count, 0) + versionAndCounts.reduce((acc, item) => acc + item.alreadyUpgrading, 0)) / totalActiveAgents * 100;
    const totalNeededPercentage = versionAndCounts.reduce((acc, item) => acc + item.targetPercentage, 0);

    // Now we have the total percentage after we add everything up, vs the total target percentage we have. Get the difference, then multiply that by the total active agents to get the delta we need to add or remove from the total count.
    const totalDeltaPercentage = totalActualPercentage - totalNeededPercentage;

    // If we are over, we need to remove some from the count, and if we are under, we need to add some to the count. If we're spot on, all good.
    if (totalDeltaPercentage !== 0) {
      // get the actual count of agents we are off by using the percentage * the total active agents
      let deltaCount = Math.round(totalDeltaPercentage / 100 * totalActiveAgents);

      // Now we need to add or remove from the versionAndCounts array
      let index = 0;
      // So long as we have more to add or remove, do so
      while (deltaCount !== 0 && index < versionAndCounts.length) {
        const item = versionAndCounts[index];
        if (deltaCount > 0) {
          // Still have too many, removing one
          item.count -= 1;
          deltaCount -= 1;
        } else if (deltaCount < 0) {
          // Still have too few, adding one
          if (item.count > 0) {
            item.count += 1;
            deltaCount += 1;
          }
        }
        index++;
      }
    }
    return versionAndCounts;
  }
  async getAgentCount(esClient, soClient, kuery) {
    const res = await (0, _agents.getAgentsByKuery)(esClient, soClient, {
      showInactive: false,
      perPage: 0,
      kuery
    });
    return res.total;
  }
  async processRequiredVersion(esClient, soClient, agentPolicy, requiredVersion, versionAndCounts) {
    var _versionAndCounts$fin, _versionAndCounts$fin2;
    this.logger.debug(`[AutomaticAgentUpgradeTask] Agent policy ${agentPolicy.id}: checking candidate agents for upgrade (target version: ${requiredVersion.version}, percentage: ${requiredVersion.percentage})`);
    let numberOfAgentsForUpgrade = (_versionAndCounts$fin = (_versionAndCounts$fin2 = versionAndCounts.find(item => item.version === requiredVersion.version)) === null || _versionAndCounts$fin2 === void 0 ? void 0 : _versionAndCounts$fin2.count) !== null && _versionAndCounts$fin !== void 0 ? _versionAndCounts$fin : 0;
    // Return if target is already met.
    if (numberOfAgentsForUpgrade <= 0) {
      this.logger.info(`[AutomaticAgentUpgradeTask] Agent policy ${agentPolicy.id}: target percentage ${requiredVersion.percentage} already reached for version: ${requiredVersion.version})`);
      return;
    }

    // Handle retries.
    const numberOfRetriedAgents = await this.processRetries(esClient, soClient, agentPolicy, requiredVersion.version);
    numberOfAgentsForUpgrade -= numberOfRetriedAgents;
    if (numberOfAgentsForUpgrade <= 0) {
      this.logger.debug(`[AutomaticAgentUpgradeTask] Number of agents ${numberOfAgentsForUpgrade}: no candidate agents found for upgrade (target version: ${requiredVersion.version}, percentage: ${requiredVersion.percentage})`);
      return;
    }

    // Fetch candidate agents assigned to the policy in batches.
    // NB: ideally, we would query active agents on or below the target version. Unfortunately, this is not possible because agent.version
    //     is stored as text, so semver comparison cannot be done in the ES query (cf. https://github.com/elastic/kibana/issues/168604).
    //     As an imperfect alternative, sort agents by version. Since versions sort alphabetically, this will not always result in ascending semver sorting.
    const statusKuery = '(status:online OR status:offline OR status:enrolling OR status:degraded OR status:error OR status:orphaned)'; // active status except updating
    const oldStuckInUpdatingKuery = `(NOT upgrade_details:* AND status:updating AND NOT upgraded_at:* AND upgrade_started_at < now-2h)`; // agents pre 8.12.0 (without upgrade_details)
    const newStuckInUpdatingKuery = `(upgrade_details.target_version:${requiredVersion.version} AND upgrade_details.state:UPG_FAILED)`;
    const agentsFetcher = await (0, _agents.fetchAllAgentsByKuery)(esClient, soClient, {
      kuery: `policy_id:${agentPolicy.id} AND (NOT upgrade_attempts:*) AND (${statusKuery} OR ${oldStuckInUpdatingKuery} OR ${newStuckInUpdatingKuery})`,
      perPage: AGENTS_BATCHSIZE,
      sortField: 'agent.version',
      sortOrder: 'asc'
    });
    let {
      done,
      agents
    } = await this.getNextAgentsBatch(agentsFetcher);
    if (agents.length === 0) {
      this.logger.debug(`[AutomaticAgentUpgradeTask] Agent policy ${agentPolicy.id}: no candidate agents found for upgrade (target version: ${requiredVersion.version}, percentage: ${requiredVersion.percentage})`);
      return;
    }
    let shouldProcessAgents = true;
    while (shouldProcessAgents) {
      this.throwIfAborted();
      numberOfAgentsForUpgrade = await this.findAndUpgradeCandidateAgents(esClient, soClient, agentPolicy, numberOfAgentsForUpgrade, requiredVersion.version, agents);
      if (!done && numberOfAgentsForUpgrade > 0) {
        ({
          done,
          agents
        } = await this.getNextAgentsBatch(agentsFetcher));
        if (done) {
          shouldProcessAgents = false;
        }
      } else {
        shouldProcessAgents = false;
      }
    }
    if (numberOfAgentsForUpgrade > 0) {
      this.logger.info(`[AutomaticAgentUpgradeTask] Agent policy ${agentPolicy.id}: not enough agents eligible for upgrade (target version: ${requiredVersion.version}, percentage: ${requiredVersion.percentage})`);
    }
  }
  async processRetries(esClient, soClient, agentPolicy, version) {
    let retriedAgentsCounter = 0;
    const retryingAgentsFetcher = await (0, _agents.fetchAllAgentsByKuery)(esClient, soClient, {
      kuery: `policy_id:${agentPolicy.id} AND upgrade_details.target_version:${version} AND upgrade_details.state:UPG_FAILED AND upgrade_attempts:*`,
      perPage: AGENTS_BATCHSIZE,
      sortField: 'agent.version',
      sortOrder: 'asc'
    });
    for await (const retryingAgentsPageResults of retryingAgentsFetcher) {
      this.throwIfAborted();
      // This function will return the total number of agents marked for retry so they're included in the count of agents for upgrade.
      retriedAgentsCounter += retryingAgentsPageResults.length;
      const agentsReadyForRetry = retryingAgentsPageResults.filter(agent => this.isAgentReadyForRetry(agent, agentPolicy));
      if (agentsReadyForRetry.length > 0) {
        this.logger.info(`[AutomaticAgentUpgradeTask] Agent policy ${agentPolicy.id}: retrying upgrade to ${version} for ${agentsReadyForRetry.length} agents`);
        await (0, _agents.sendAutomaticUpgradeAgentsActions)(soClient, esClient, {
          agents: agentsReadyForRetry,
          version,
          spaceIds: agentPolicy.space_ids,
          ...this.getUpgradeDurationSeconds(agentsReadyForRetry.length)
        });
      }
    }
    return retriedAgentsCounter;
  }
  isAgentReadyForRetry(agent, agentPolicy) {
    if (!agent.upgrade_attempts) {
      return false;
    }
    if (agent.upgrade_attempts.length > this.retryDelays.length) {
      this.logger.debug(`[AutomaticAgentUpgradeTask] Agent policy ${agentPolicy.id}: max retry attempts exceeded for agent ${agent.id}`);
      return false;
    }
    const currentRetryDelay = _moment.default.duration('PT' + this.retryDelays[agent.upgrade_attempts.length - 1].toUpperCase()) // https://momentjs.com/docs/#/durations/
    .asMilliseconds();
    const lastUpgradeAttempt = Date.parse(agent.upgrade_attempts[0]);
    return Date.now() - lastUpgradeAttempt >= currentRetryDelay;
  }
  async getNextAgentsBatch(agentsFetcher) {
    var _agentsBatch$value;
    const agentsFetcherIter = agentsFetcher[Symbol.asyncIterator]();
    const agentsBatch = await agentsFetcherIter.next();
    const agents = (_agentsBatch$value = agentsBatch.value) !== null && _agentsBatch$value !== void 0 ? _agentsBatch$value : [];
    return {
      done: agentsBatch.done,
      agents: agents.filter(agent => agent.agent !== undefined)
    };
  }
  async findAndUpgradeCandidateAgents(esClient, soClient, agentPolicy, numberOfAgentsForUpgrade, version, agents) {
    const agentsForUpgrade = [];
    for (const agent of agents) {
      if (agentsForUpgrade.length >= numberOfAgentsForUpgrade) {
        break;
      }
      if (this.isAgentEligibleForUpgrade(agent, version)) {
        agentsForUpgrade.push(agent);
      }
    }

    // Send bulk upgrade action for selected agents.
    if (agentsForUpgrade.length > 0) {
      this.logger.info(`[AutomaticAgentUpgradeTask] Agent policy ${agentPolicy.id}: sending bulk upgrade to ${version} for ${agentsForUpgrade.length} agents`);
      await (0, _agents.sendAutomaticUpgradeAgentsActions)(soClient, esClient, {
        agents: agentsForUpgrade,
        version,
        spaceIds: agentPolicy.space_ids,
        ...this.getUpgradeDurationSeconds(agentsForUpgrade.length)
      });
    }
    return numberOfAgentsForUpgrade - agentsForUpgrade.length;
  }
  isAgentEligibleForUpgrade(agent, version) {
    return (0, _services2.isAgentUpgradeable)(agent) && (0, _gt.default)(version, agent.agent.version);
  }
  getUpgradeDurationSeconds(nAgents) {
    if (nAgents < MIN_AGENTS_FOR_ROLLOUT) {
      return {};
    }
    const upgradeDurationSeconds = Math.max(MIN_UPGRADE_DURATION_SECONDS, Math.round(nAgents * 0.03));
    return {
      upgradeDurationSeconds
    };
  }
}
exports.AutomaticAgentUpgradeTask = AutomaticAgentUpgradeTask;