"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postDefendInsightsRoute = void 0;
var _moment = _interopRequireDefault(require("moment/moment"));
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _build_response = require("../../lib/build_response");
var _helpers = require("./helpers");
var _app_context = require("../../services/app_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ROUTE_HANDLER_TIMEOUT = 10 * 60 * 1000; // 10 * 60 seconds = 10 minutes
const LANG_CHAIN_TIMEOUT = ROUTE_HANDLER_TIMEOUT - 10_000; // 9 minutes 50 seconds
const CONNECTOR_TIMEOUT = LANG_CHAIN_TIMEOUT - 10_000; // 9 minutes 40 seconds

const postDefendInsightsRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _elasticAssistantCommon.DEFEND_INSIGHTS,
    options: {
      timeout: {
        idleSocket: ROUTE_HANDLER_TIMEOUT
      }
    },
    security: {
      authz: {
        requiredPrivileges: ['securitySolution-writeWorkflowInsights']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.DefendInsightsPostRequestBody)
      },
      response: {
        200: {
          body: {
            custom: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.DefendInsightsPostResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const startTime = (0, _moment.default)(); // start timing the generation
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['licensing', 'elasticAssistant']);
    const assistantContext = ctx.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    const savedObjectsClient = assistantContext.savedObjectsClient;
    try {
      const isEnabled = (0, _helpers.isDefendInsightsEnabled)({
        request,
        logger,
        assistantContext
      });
      if (!isEnabled) {
        return response.notFound();
      }
      if (!ctx.licensing.license.hasAtLeast('enterprise')) {
        return response.forbidden({
          body: {
            message: 'Your license does not support Defend Workflows. Please upgrade your license.'
          }
        });
      }
      const actions = assistantContext.actions;
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const dataClient = await assistantContext.getDefendInsightsDataClient();
      const authenticatedUser = await assistantContext.getCurrentUser();
      if (authenticatedUser == null) {
        return resp.error({
          body: `Authenticated user not found`,
          statusCode: 401
        });
      }
      if (!dataClient) {
        return resp.error({
          body: `Defend insights data client not initialized`,
          statusCode: 500
        });
      }
      await (0, _helpers.runExternalCallbacks)(_app_context.CallbackIds.DefendInsightsPreCreate, request);
      const {
        endpointIds,
        insightType,
        apiConfig,
        anonymizationFields,
        langSmithApiKey,
        langSmithProject,
        replacements
      } = request.body;
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      let latestReplacements = {
        ...replacements
      };
      const onNewReplacements = newReplacements => {
        latestReplacements = {
          ...latestReplacements,
          ...newReplacements
        };
      };
      const {
        currentInsight,
        defendInsightId
      } = await (0, _helpers.createDefendInsight)(endpointIds, insightType, dataClient, authenticatedUser, apiConfig);
      (0, _helpers.invokeDefendInsightsGraph)({
        insightType,
        endpointIds,
        actionsClient,
        anonymizationFields,
        apiConfig,
        connectorTimeout: CONNECTOR_TIMEOUT,
        esClient,
        langSmithProject,
        langSmithApiKey,
        latestReplacements,
        logger,
        onNewReplacements,
        savedObjectsClient
      }).then(({
        anonymizedEvents,
        insights
      }) => (0, _helpers.updateDefendInsights)({
        anonymizedEvents,
        apiConfig,
        defendInsightId,
        insights,
        authenticatedUser,
        dataClient,
        latestReplacements,
        logger,
        startTime,
        telemetry,
        insightType
      }).then(() => insights)).then(insights => _app_context.appContextService.getRegisteredCallbacks(_app_context.CallbackIds.DefendInsightsPostCreate).map(cb => cb(insights, request))).catch(err => (0, _helpers.handleGraphError)({
        apiConfig,
        defendInsightId,
        authenticatedUser,
        dataClient,
        err,
        latestReplacements,
        logger,
        telemetry
      }));
      return response.ok({
        body: currentInsight
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postDefendInsightsRoute = postDefendInsightsRoute;