"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeReduxSync = initializeReduxSync;
var _rxjs = require("rxjs");
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _constants = require("../../common/constants");
var _ui_selectors = require("../selectors/ui_selectors");
var _map_selectors = require("../selectors/map_selectors");
var _actions = require("../actions");
var _kibana_services = require("../kibana_services");
var _non_serializable_instances = require("../reducers/non_serializable_instances");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getMapCenterAndZoom(state) {
  return {
    ...(0, _map_selectors.getMapCenter)(state),
    zoom: (0, _map_selectors.getMapZoom)(state)
  };
}
function getHiddenLayerIds(state) {
  return (0, _map_selectors.getLayerListRaw)(state).filter(layer => !layer.visible).map(layer => layer.id);
}
function initializeReduxSync({
  savedMap,
  state,
  syncColors$,
  uuid
}) {
  var _state$hiddenLayers, _state$isLayerTOCOpen, _state$mapCenter, _state$openTOCDetails;
  const store = savedMap.getStore();

  // initializing comparitor publishing subjects to state instead of store state values
  // because store is not settled until map is rendered and mapReady is true
  const hiddenLayers$ = new _rxjs.BehaviorSubject((_state$hiddenLayers = state.hiddenLayers) !== null && _state$hiddenLayers !== void 0 ? _state$hiddenLayers : getHiddenLayerIds(store.getState()));
  const isLayerTOCOpen$ = new _rxjs.BehaviorSubject((_state$isLayerTOCOpen = state.isLayerTOCOpen) !== null && _state$isLayerTOCOpen !== void 0 ? _state$isLayerTOCOpen : (0, _ui_selectors.getIsLayerTOCOpen)(store.getState()));
  const mapCenterAndZoom$ = new _rxjs.BehaviorSubject((_state$mapCenter = state.mapCenter) !== null && _state$mapCenter !== void 0 ? _state$mapCenter : getMapCenterAndZoom(store.getState()));
  const openTOCDetails$ = new _rxjs.BehaviorSubject((_state$openTOCDetails = state.openTOCDetails) !== null && _state$openTOCDetails !== void 0 ? _state$openTOCDetails : (0, _ui_selectors.getOpenTOCDetails)(store.getState()));
  const dataLoading$ = new _rxjs.BehaviorSubject(undefined);
  const unsubscribeFromStore = store.subscribe(() => {
    if (!(0, _map_selectors.getMapReady)(store.getState())) {
      return;
    }
    const nextHiddenLayers = getHiddenLayerIds(store.getState());
    if (!(0, _fastDeepEqual.default)(hiddenLayers$.value, nextHiddenLayers)) {
      hiddenLayers$.next(nextHiddenLayers);
    }
    const nextIsLayerTOCOpen = (0, _ui_selectors.getIsLayerTOCOpen)(store.getState());
    if (isLayerTOCOpen$.value !== nextIsLayerTOCOpen) {
      isLayerTOCOpen$.next(nextIsLayerTOCOpen);
    }
    const nextMapCenterAndZoom = getMapCenterAndZoom(store.getState());
    if (!(0, _fastDeepEqual.default)(mapCenterAndZoom$.value, nextMapCenterAndZoom)) {
      mapCenterAndZoom$.next(nextMapCenterAndZoom);
    }
    const nextOpenTOCDetails = (0, _ui_selectors.getOpenTOCDetails)(store.getState());
    if (!(0, _fastDeepEqual.default)(openTOCDetails$.value, nextOpenTOCDetails)) {
      openTOCDetails$.next(nextOpenTOCDetails);
    }
    const nextIsMapLoading = (0, _map_selectors.isMapLoading)(store.getState());
    if (nextIsMapLoading !== dataLoading$.value) {
      dataLoading$.next(nextIsMapLoading);
    }
  });
  store.dispatch((0, _actions.setReadOnly)(true));
  store.dispatch((0, _actions.setMapSettings)({
    keydownScrollZoom: true,
    showTimesliderToggleButton: false
  }));
  store.dispatch((0, _actions.setExecutionContext)(getExecutionContext(uuid, state.savedObjectId)));
  const filters$ = new _rxjs.BehaviorSubject(undefined);
  const query$ = new _rxjs.BehaviorSubject(undefined);
  const mapStateJSON = savedMap.getAttributes().mapStateJSON;
  if (mapStateJSON) {
    try {
      const mapState = JSON.parse(mapStateJSON);
      if (mapState.filters) {
        filters$.next(mapState.filters);
      }
      if (mapState.query) {
        query$.next(mapState.query);
      }
      store.dispatch((0, _actions.setEmbeddableSearchContext)({
        filters: mapState.filters,
        query: mapState.query
      }));
    } catch (e) {
      // ignore malformed mapStateJSON, not a critical error for viewing map - map will just use defaults
    }
  }
  let syncColorsSubscription;
  let syncColorsSymbol;
  if (syncColors$) {
    syncColorsSubscription = syncColors$.subscribe(async syncColors => {
      const currentSyncColorsSymbol = Symbol();
      syncColorsSymbol = currentSyncColorsSymbol;
      const chartsPaletteServiceGetColor = syncColors ? await getChartsPaletteServiceGetColor() : null;
      if (syncColorsSymbol === currentSyncColorsSymbol) {
        store.dispatch((0, _non_serializable_instances.setChartsPaletteServiceGetColor)(chartsPaletteServiceGetColor));
      }
    });
  }
  return {
    cleanup: () => {
      if (syncColorsSubscription) syncColorsSubscription.unsubscribe();
      unsubscribeFromStore();
    },
    api: {
      dataLoading$,
      filters$,
      getInspectorAdapters: () => {
        return (0, _non_serializable_instances.getInspectorAdapters)(store.getState());
      },
      getLayerList: () => {
        return (0, _map_selectors.getLayerList)(store.getState());
      },
      onRenderComplete$: dataLoading$.pipe((0, _rxjs.filter)(isDataLoading => typeof isDataLoading === 'boolean' && !isDataLoading), (0, _rxjs.debounceTime)(_constants.RENDER_TIMEOUT), (0, _rxjs.map)(() => {
        // Observable notifies subscriber when rendering is complete
        // Return void to not expose internal implemenation details of observabale
        return;
      })),
      query$,
      reload: () => {
        store.dispatch((0, _actions.setQuery)({
          forceRefresh: true
        }));
      },
      setEventHandlers: eventHandlers => {
        store.dispatch((0, _non_serializable_instances.setEventHandlers)(eventHandlers));
      }
    },
    comparators: {
      // mapBuffer comparator intentionally omitted and is not part of unsaved changes check
      hiddenLayers: [hiddenLayers$, nextValue => {
        store.dispatch((0, _actions.setHiddenLayers)(nextValue));
      }, _fastDeepEqual.default],
      isLayerTOCOpen: [isLayerTOCOpen$, nextValue => {
        store.dispatch((0, _actions.setIsLayerTOCOpen)(nextValue));
      }],
      mapCenter: [mapCenterAndZoom$, nextValue => {
        store.dispatch((0, _actions.setGotoWithCenter)(nextValue));
      }, _fastDeepEqual.default],
      openTOCDetails: [openTOCDetails$, nextValue => {
        store.dispatch((0, _actions.setOpenTOCDetails)(nextValue));
      }, _fastDeepEqual.default]
    },
    serialize: () => {
      return {
        hiddenLayers: getHiddenLayerIds(store.getState()),
        isLayerTOCOpen: (0, _ui_selectors.getIsLayerTOCOpen)(store.getState()),
        mapBuffer: (0, _map_selectors.getMapBuffer)(store.getState()),
        mapCenter: getMapCenterAndZoom(store.getState()),
        openTOCDetails: (0, _ui_selectors.getOpenTOCDetails)(store.getState())
      };
    }
  };
}
function getExecutionContext(uuid, savedObjectId) {
  const parentContext = (0, _kibana_services.getExecutionContextService)().get();
  const mapContext = {
    type: _constants.APP_ID,
    name: _constants.APP_ID,
    id: uuid,
    url: (0, _constants.getEditPath)(savedObjectId)
  };
  return parentContext ? {
    ...parentContext,
    child: mapContext
  } : mapContext;
}
async function getChartsPaletteServiceGetColor() {
  const chartsService = (0, _kibana_services.getCharts)();
  const paletteRegistry = chartsService ? await chartsService.palettes.getPalettes() : null;
  if (!paletteRegistry) {
    return null;
  }
  const paletteDefinition = paletteRegistry.get('default');
  const chartConfiguration = {
    syncColors: true
  };
  return value => {
    const series = [{
      name: value,
      rankAtDepth: 0,
      totalSeriesAtDepth: 1
    }];
    const color = paletteDefinition.getCategoricalColor(series, chartConfiguration);
    return color ? color : '#3d3d3d';
  };
}