"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ActionsClientChatOpenAI = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _fp = require("lodash/fp");
var _openai = require("@langchain/openai");
var _constants = require("./constants");
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _classPrivateFieldSet(s, a, r) { return s.set(_assertClassBrand(s, a), r), r; }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const LLM_TYPE = 'ActionsClientChatOpenAI';
var _temperature = /*#__PURE__*/new WeakMap();
var _actionsClient = /*#__PURE__*/new WeakMap();
var _connectorId = /*#__PURE__*/new WeakMap();
var _logger = /*#__PURE__*/new WeakMap();
var _actionResultData = /*#__PURE__*/new WeakMap();
var _traceId = /*#__PURE__*/new WeakMap();
var _signal = /*#__PURE__*/new WeakMap();
var _timeout = /*#__PURE__*/new WeakMap();
/**
 * This class is a wrapper around the ChatOpenAI class from @langchain/openai.
 * It is used to call the OpenAI API via the Actions plugin.
 * It is used by the OpenAI connector type only.
 * The completionWithRetry method is overridden to use the Actions plugin.
 * In the ChatOpenAI class, *_streamResponseChunks calls completionWithRetry
 * and iterates over the chunks to form the response.
 */
class ActionsClientChatOpenAI extends _openai.ChatOpenAI {
  constructor({
    actionsClient,
    connectorId,
    traceId = (0, _uuid.v4)(),
    llmType,
    logger,
    maxRetries,
    model,
    signal,
    streaming = true,
    temperature,
    timeout,
    maxTokens,
    telemetryMetadata
  }) {
    super({
      maxRetries,
      maxTokens,
      streaming,
      // matters only for the LangSmith logs (Metadata > Invocation Params), which are misleading if this is not set
      modelName: model !== null && model !== void 0 ? model : _constants.DEFAULT_OPEN_AI_MODEL,
      openAIApiKey: ''
    });
    (0, _defineProperty2.default)(this, "streaming", void 0);
    // Local `llmType` as it can change and needs to be accessed by abstract `_llmType()` method
    // Not using getter as `this._llmType()` is called in the constructor via `super({})`
    (0, _defineProperty2.default)(this, "llmType", void 0);
    // ChatOpenAI class needs these, but they do not matter as we override the openai client with the actions client
    (0, _defineProperty2.default)(this, "azureOpenAIApiKey", '');
    (0, _defineProperty2.default)(this, "openAIApiKey", '');
    (0, _defineProperty2.default)(this, "model", void 0);
    _classPrivateFieldInitSpec(this, _temperature, void 0);
    // Kibana variables
    _classPrivateFieldInitSpec(this, _actionsClient, void 0);
    _classPrivateFieldInitSpec(this, _connectorId, void 0);
    _classPrivateFieldInitSpec(this, _logger, void 0);
    _classPrivateFieldInitSpec(this, _actionResultData, void 0);
    _classPrivateFieldInitSpec(this, _traceId, void 0);
    _classPrivateFieldInitSpec(this, _signal, void 0);
    _classPrivateFieldInitSpec(this, _timeout, void 0);
    (0, _defineProperty2.default)(this, "telemetryMetadata", void 0);
    _classPrivateFieldSet(_actionsClient, this, actionsClient);
    _classPrivateFieldSet(_connectorId, this, connectorId);
    _classPrivateFieldSet(_traceId, this, traceId);
    this.llmType = llmType !== null && llmType !== void 0 ? llmType : LLM_TYPE;
    _classPrivateFieldSet(_logger, this, logger);
    _classPrivateFieldSet(_timeout, this, timeout);
    _classPrivateFieldSet(_actionResultData, this, '');
    this.streaming = streaming;
    _classPrivateFieldSet(_signal, this, signal);
    this.model = model !== null && model !== void 0 ? model : _constants.DEFAULT_OPEN_AI_MODEL;
    // to be passed to the actions client
    _classPrivateFieldSet(_temperature, this, temperature);
    // matters only for LangSmith logs (Metadata > Invocation Params)
    // the connector can be passed an undefined temperature through #temperature
    this.temperature = temperature !== null && temperature !== void 0 ? temperature : this.temperature;
    this.telemetryMetadata = telemetryMetadata;
  }
  getActionResultData() {
    return _classPrivateFieldGet(_actionResultData, this);
  }
  _llmType() {
    return this.llmType;
  }

  // Model type needs to be `base_chat_model` to work with LangChain OpenAI Tools
  // We may want to make this configurable (ala _llmType) if different agents end up requiring different model types
  // See: https://github.com/langchain-ai/langchainjs/blob/fb699647a310c620140842776f4a7432c53e02fa/langchain/src/agents/openai/index.ts#L185
  _modelType() {
    return 'base_chat_model';
  }
  async completionWithRetry(completionRequest) {
    return this.caller.call(async () => {
      const requestBody = this.formatRequestForActionsClient(completionRequest, this.llmType);
      _classPrivateFieldGet(_logger, this).debug(() => `${LLM_TYPE}#completionWithRetry ${_classPrivateFieldGet(_traceId, this)} assistantMessage:\n${JSON.stringify(requestBody.params.subActionParams)} `);
      const actionResult = await _classPrivateFieldGet(_actionsClient, this).execute(requestBody);
      if (actionResult.status === 'error') {
        const error = new Error(`${LLM_TYPE}: ${actionResult === null || actionResult === void 0 ? void 0 : actionResult.message} - ${actionResult === null || actionResult === void 0 ? void 0 : actionResult.serviceMessage}`);
        if (actionResult !== null && actionResult !== void 0 && actionResult.serviceMessage) {
          error.name = actionResult === null || actionResult === void 0 ? void 0 : actionResult.serviceMessage;
        }
        throw error;
      }
      if (!this.streaming) {
        // typecasting as the `run` subaction returns the OpenAI.ChatCompletion directly from OpenAI
        const chatCompletion = (0, _fp.get)('data', actionResult);
        return chatCompletion;
      }

      // cast typing as this is the contract of the actions client
      const result = (0, _fp.get)('data', actionResult);
      if (result.consumerStream == null) {
        throw new Error(`${LLM_TYPE}: action result data is empty ${actionResult}`);
      }
      return result.consumerStream;
    });
  }
  formatRequestForActionsClient(completionRequest, llmType) {
    var _classPrivateFieldGet2, _classPrivateFieldGet3;
    const body = {
      temperature: _classPrivateFieldGet(_temperature, this),
      // possible client model override
      // security sends this from connectors, it is only missing from preconfigured connectors
      // this should be undefined otherwise so the connector handles the model (stack_connector has access to preconfigured connector model values)
      ...(llmType === 'inference' ? {} : {
        model: this.model
      }),
      n: completionRequest.n,
      stop: completionRequest.stop,
      tools: completionRequest.tools,
      ...(completionRequest.tool_choice ? {
        tool_choice: completionRequest.tool_choice
      } : {}),
      // deprecated, use tools
      ...(completionRequest.functions ? {
        functions: completionRequest === null || completionRequest === void 0 ? void 0 : completionRequest.functions
      } : {}),
      // ensure we take the messages from the completion request, not the client request
      messages: completionRequest.messages.map(message => {
        var _message$content;
        return {
          role: message.role,
          content: (_message$content = message.content) !== null && _message$content !== void 0 ? _message$content : '',
          ...('name' in message ? {
            name: message === null || message === void 0 ? void 0 : message.name
          } : {}),
          ...('tool_calls' in message ? {
            tool_calls: message === null || message === void 0 ? void 0 : message.tool_calls
          } : {}),
          ...('tool_call_id' in message ? {
            tool_call_id: message === null || message === void 0 ? void 0 : message.tool_call_id
          } : {}),
          // deprecated, use tool_calls
          ...('function_call' in message ? {
            function_call: message === null || message === void 0 ? void 0 : message.function_call
          } : {})
        };
      })
    };
    const subAction = llmType === 'inference' ? completionRequest.stream ? 'unified_completion_async_iterator' : 'unified_completion' :
    // langchain expects stream to be of type AsyncIterator<OpenAI.ChatCompletionChunk>
    // for non-stream, use `run` instead of `invokeAI` in order to get the entire OpenAI.ChatCompletion response,
    // which may contain non-content messages like functions
    completionRequest.stream ? 'invokeAsyncIterator' : 'run';
    // create a new connector request body with the assistant message:
    const subActionParams = {
      ...(llmType === 'inference' ? {
        body
      } : completionRequest.stream ? {
        ...body,
        timeout: (_classPrivateFieldGet2 = _classPrivateFieldGet(_timeout, this)) !== null && _classPrivateFieldGet2 !== void 0 ? _classPrivateFieldGet2 : _constants.DEFAULT_TIMEOUT
      } : {
        body: JSON.stringify(body),
        timeout: (_classPrivateFieldGet3 = _classPrivateFieldGet(_timeout, this)) !== null && _classPrivateFieldGet3 !== void 0 ? _classPrivateFieldGet3 : _constants.DEFAULT_TIMEOUT
      }),
      telemetryMetadata: this.telemetryMetadata,
      signal: _classPrivateFieldGet(_signal, this)
    };
    return {
      actionId: _classPrivateFieldGet(_connectorId, this),
      params: {
        subAction,
        subActionParams
      },
      signal: _classPrivateFieldGet(_signal, this)
    };
  }
}
exports.ActionsClientChatOpenAI = ActionsClientChatOpenAI;