"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEntitiesWithSource = getEntitiesWithSource;
var _create_entities_es_client = require("../clients/create_entities_es_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// the official types do not explicitly define sourceIndex in the schema, but it is present in the data at the time of writing this

async function getEntitiesWithSource({
  serviceEnvironment,
  serviceName,
  containerId,
  hostName,
  entitiesEsClient,
  esClient
}) {
  const entityCategoryPromises = getFetchEntitiesPromises({
    entitiesEsClient,
    serviceName,
    serviceEnvironment,
    hostName,
    containerId
  });
  const entityResponses = await Promise.all(entityCategoryPromises);
  const entitiesWithSource = [];
  for (const response of entityResponses) {
    const processedEntities = await Promise.all(response.map(async entity => {
      const sourceIndex = entity === null || entity === void 0 ? void 0 : entity.source_index;
      if (!sourceIndex || !sourceIndex.length) return null;
      const indices = await esClient.indices.get({
        index: sourceIndex
      });
      const sources = await fetchSources(indices);
      return {
        identity_fields: entity === null || entity === void 0 ? void 0 : entity.entity.identity_fields,
        id: entity === null || entity === void 0 ? void 0 : entity.entity.id,
        definition_id: entity === null || entity === void 0 ? void 0 : entity.entity.definition_id,
        last_seen_timestamp: entity === null || entity === void 0 ? void 0 : entity.entity.last_seen_timestamp,
        display_name: entity === null || entity === void 0 ? void 0 : entity.entity.display_name,
        metrics: entity === null || entity === void 0 ? void 0 : entity.entity.metrics,
        schema_version: entity === null || entity === void 0 ? void 0 : entity.entity.schema_version,
        definition_version: entity === null || entity === void 0 ? void 0 : entity.entity.definition_version,
        type: entity === null || entity === void 0 ? void 0 : entity.entity.type,
        sources
      };
    }));
    entitiesWithSource.push(...processedEntities.filter(Boolean));
  }
  return {
    entities: entitiesWithSource
  };
}
function fetchSources(indices) {
  return Object.values(indices).map(index => {
    return getEntitySource({
      index
    });
  });
}
const getEntitySource = ({
  index
}) => {
  const dataStream = index.data_stream;
  const source = {
    dataStream
  };
  return source;
};
const getFetchEntitiesPromises = ({
  entitiesEsClient,
  serviceName,
  serviceEnvironment,
  hostName,
  containerId
}) => {
  const shouldFilterForServiceEnvironment = serviceEnvironment && serviceName && serviceEnvironment !== 'ENVIRONMENT_ALL' && serviceEnvironment !== 'ENVIRONMENT_NOT_DEFINED';
  const containersPromise = getFetchEntityPromise({
    index: _create_entities_es_client.CONTAINER_ENTITIES_LATEST_ALIAS,
    shouldFetch: Boolean(hostName || containerId),
    shouldMatch: [...(hostName ? [{
      term: {
        'host.name': hostName
      }
    }] : []), ...(containerId ? [{
      term: {
        'container.id': containerId
      }
    }] : [])],
    entitiesEsClient
  });
  const hostsPromise = getFetchEntityPromise({
    index: _create_entities_es_client.HOST_ENTITIES_LATEST_ALIAS,
    shouldFetch: Boolean(hostName),
    shouldMatch: hostName ? [{
      term: {
        'host.name': hostName
      }
    }] : [],
    entitiesEsClient
  });
  const servicesPromise = getFetchEntityPromise({
    index: _create_entities_es_client.SERVICE_ENTITIES_LATEST_ALIAS,
    shouldFetch: Boolean(serviceName),
    shouldMatch: [...(serviceName ? [{
      term: {
        'service.name': serviceName
      }
    }] : []), ...(shouldFilterForServiceEnvironment ? [{
      term: {
        'service.environment': serviceEnvironment
      }
    }] : [])],
    entitiesEsClient
  });
  return [containersPromise, hostsPromise, servicesPromise].filter(promise => promise !== null);
};
const getFetchEntityPromise = ({
  index,
  shouldFetch,
  shouldMatch,
  entitiesEsClient
}) => {
  return shouldFetch ? entitiesEsClient.search(index, {
    body: {
      query: {
        bool: {
          should: shouldMatch,
          minimum_should_match: 1
        }
      }
    }
  }).then(response => {
    return response.hits.hits.map(hit => {
      return {
        source_index: hit === null || hit === void 0 ? void 0 : hit._source.source_index,
        entity: hit._source.entity
      };
    });
  }) : null;
};