"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.from = exports.arrayOfStrings = exports.EDITOR_PX_HEIGHT = void 0;
exports.isArrayOfStrings = isArrayOfStrings;
exports.to = exports.isXJsonValue = exports.isXJsonField = exports.isJSONStringValidator = exports.isEmptyString = void 0;
var rt = _interopRequireWildcard(require("io-ts"));
var _i18n = require("@kbn/i18n");
var _Either = require("fp-ts/lib/Either");
var _lodash = require("lodash");
var _shared_imports = require("../../../../../../shared_imports");
var _utils = require("../../../utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const {
  emptyField,
  isJsonField
} = _shared_imports.fieldValidators;
const arrayOfStrings = exports.arrayOfStrings = rt.array(rt.string);
function isArrayOfStrings(v) {
  const res = arrayOfStrings.decode(v);
  return (0, _Either.isRight)(res);
}

/**
 * Format a XJson string input as parsed JSON. Replaces the invalid characters
 *  with a placeholder, parses the new string in a JSON format with the expected
 * indentantion and then replaces the placeholders with the original values.
 */
const formatXJsonString = input => {
  let placeholder = 'PLACEHOLDER';
  const INVALID_STRING_REGEX = /"""(.*?)"""/gs;
  while (input.includes(placeholder)) {
    placeholder += '_';
  }
  const modifiedInput = input.replace(INVALID_STRING_REGEX, () => `"${placeholder}"`);
  let jsonObject;
  try {
    jsonObject = JSON.parse(modifiedInput);
  } catch (error) {
    return input;
  }
  let formattedJsonString = JSON.stringify(jsonObject, null, 2);
  const invalidStrings = input.match(INVALID_STRING_REGEX);
  if (invalidStrings) {
    invalidStrings.forEach(invalidString => {
      formattedJsonString = formattedJsonString.replace(`"${placeholder}"`, invalidString);
    });
  }
  return formattedJsonString;
};

/**
 * Shared deserializer functions.
 *
 * These are intended to be used in @link{FieldsConfig} as the "deserializer".
 *
 * Example:
 * {
 *   ...
 *   deserialize: to.booleanOrUndef,
 *   ...
 * }
 *
 */
const to = exports.to = {
  booleanOrUndef: v => typeof v === 'boolean' ? v : undefined,
  arrayOfStrings: v => isArrayOfStrings(v) ? v : typeof v === 'string' && v.length ? [v] : [],
  jsonString: v => v ? JSON.stringify(v, null, 2) : '{}',
  /**
   * Useful when deserializing strings that will be rendered inside of text areas or text inputs. We want
   * a string like: "my\ttab" to render the same, not to render as "my<tab>tab".
   */
  escapeBackslashes: v => {
    if (typeof v === 'string') {
      const s = JSON.stringify(v);
      return s.slice(1, s.length - 1);
    }
    return v;
  },
  xJsonString: v => {
    if (!v) {
      return '{}';
    }
    if (typeof v === 'string') {
      return formatXJsonString(v);
    }
    return JSON.stringify(v, null, 2);
  }
};

/**
 * Shared serializer functions.
 *
 * These are intended to be used in @link{FieldsConfig} as the "serializer".
 *
 * Example:
 * {
 *   ...
 *   serializer: from.optionalJson,
 *   ...
 * }
 *
 */
const from = exports.from = {
  /* Works with `to.jsonString` as deserializer. */
  optionalJson: v => {
    if (v) {
      try {
        const json = JSON.parse(v);
        if (Object.keys(json).length) {
          return json;
        }
      } catch (e) {
        // Ignore
      }
    }
    return undefined;
  },
  optionalArrayOfStrings: v => v.length ? v : undefined,
  undefinedIfValue: value => v => v === value ? undefined : v,
  emptyStringToUndefined: v => v === '' ? undefined : v,
  /**
   * Useful when serializing user input from a <textarea /> that we want to later JSON.stringify but keep the same as what
   * the user input. For instance, given "my\ttab", encoded as "my%5Ctab" will JSON.stringify to "my\\ttab", instead we want
   * to keep the input exactly as the user entered it.
   */
  unescapeBackslashes: v => {
    if (typeof v === 'string') {
      try {
        return JSON.parse(`"${v}"`);
      } catch (e) {
        // Best effort
        return v;
      }
    }
  },
  optionalXJson: v => {
    if (v && v !== '{}') {
      return v;
    }
    return undefined;
  }
};
const isJSONString = v => {
  try {
    JSON.parse(`"${v}"`);
    return true;
  } catch (e) {
    return false;
  }
};
const isJSONStringValidator = ({
  value
}) => {
  if (typeof value !== 'string' || !isJSONString(value)) {
    return {
      message: _i18n.i18n.translate('xpack.ingestPipelines.pipelineEditor.jsonStringField.invalidStringMessage', {
        defaultMessage: 'Invalid JSON string.'
      })
    };
  }
};
exports.isJSONStringValidator = isJSONStringValidator;
const isXJsonField = (message, {
  allowEmptyString = false
} = {}) => (...args) => {
  const [{
    value,
    ...rest
  }] = args;
  return isJsonField(message, {
    allowEmptyString
  })({
    ...rest,
    value: (0, _utils.collapseEscapedStrings)(value)
  });
};

/**
 * Similar to the emptyField validator but we accept whitespace characters.
 */
exports.isXJsonField = isXJsonField;
const isEmptyString = message => field => {
  const {
    value
  } = field;
  if (typeof value === 'string') {
    const hasLength = Boolean(value.length);
    const hasNonWhiteSpaceChars = hasLength && Boolean(value.trim().length);
    if (hasNonWhiteSpaceChars) {
      return emptyField(message)(field);
    }
  }
};
exports.isEmptyString = isEmptyString;
const EDITOR_PX_HEIGHT = exports.EDITOR_PX_HEIGHT = {
  extraSmall: 75,
  small: 100,
  medium: 200,
  large: 300
};
const isXJsonValue = value => {
  if (typeof value === 'string') {
    return (0, _shared_imports.isJSON)((0, _utils.collapseEscapedStrings)(value));
  }
  return (0, _lodash.isPlainObject)(value);
};
exports.isXJsonValue = isXJsonValue;