"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeSourceConfiguration = exports.convertSavedObjectToSavedSourceConfiguration = exports.InfraSources = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _Either = require("fp-ts/lib/Either");
var _function = require("fp-ts/lib/function");
var _pipeable = require("fp-ts/lib/pipeable");
var _PathReporter = require("io-ts/lib/PathReporter");
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _ = require(".");
var _defaults = require("./defaults");
var _errors = require("./errors");
var _saved_object_references = require("./saved_object_references");
var _saved_object_type = require("./saved_object_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// extract public interface

class InfraSources {
  constructor(params) {
    (0, _defineProperty2.default)(this, "metricsClient", void 0);
    this.metricsClient = params.metricsClient;
  }
  async getInfraSourceConfiguration(savedObjectsClient, sourceId) {
    const savedSourceConfiguration = await this.getSavedSourceConfiguration(savedObjectsClient, sourceId).then(result => ({
      ...result,
      configuration: mergeSourceConfiguration(_defaults.defaultSourceConfiguration, result.configuration)
    })).catch(err => _server.SavedObjectsErrorHelpers.isNotFoundError(err) ? Promise.resolve({
      id: sourceId,
      version: undefined,
      updatedAt: undefined,
      origin: 'fallback',
      configuration: _defaults.defaultSourceConfiguration
    }) : Promise.reject(err));
    return savedSourceConfiguration;
  }
  async getSourceConfiguration(savedObjectsClient, sourceId) {
    const sourceConfiguration = await this.getInfraSourceConfiguration(savedObjectsClient, sourceId);
    const metricAlias = await this.metricsClient.getMetricIndices({
      savedObjectsClient
    });
    sourceConfiguration.configuration.metricAlias = metricAlias;
    return sourceConfiguration;
  }
  async createSourceConfiguration(savedObjectsClient, sourceId, source) {
    const {
      anomalyThreshold
    } = source;
    if (anomalyThreshold && !(0, _lodash.inRange)(anomalyThreshold, 0, 101)) throw new _errors.AnomalyThresholdRangeError('Anomaly threshold must be 1-100');
    const newSourceConfiguration = mergeSourceConfiguration(_defaults.defaultSourceConfiguration, source);
    const {
      attributes,
      references
    } = (0, _saved_object_references.extractSavedObjectReferences)(newSourceConfiguration);
    const createdSourceConfiguration = convertSavedObjectToSavedSourceConfiguration(await savedObjectsClient.create(_saved_object_type.infraSourceConfigurationSavedObjectName, attributes, {
      id: sourceId,
      references
    }));
    await this.metricsClient.updateMetricIndices({
      savedObjectsClient,
      metricIndices: newSourceConfiguration.metricAlias
    });
    return {
      ...createdSourceConfiguration,
      configuration: mergeSourceConfiguration(_defaults.defaultSourceConfiguration, createdSourceConfiguration.configuration)
    };
  }
  async deleteSourceConfiguration(savedObjectsClient, sourceId) {
    await savedObjectsClient.delete(_saved_object_type.infraSourceConfigurationSavedObjectName, sourceId);
  }
  async updateSourceConfiguration(savedObjectsClient, sourceId, sourceProperties) {
    const {
      anomalyThreshold
    } = sourceProperties;
    if (anomalyThreshold && !(0, _lodash.inRange)(anomalyThreshold, 0, 101)) throw new _errors.AnomalyThresholdRangeError('Anomaly threshold must be 1-100');
    const {
      configuration,
      version
    } = await this.getSourceConfiguration(savedObjectsClient, sourceId);
    const updatedSourceConfigurationAttributes = mergeSourceConfiguration(configuration, sourceProperties);
    const {
      attributes,
      references
    } = (0, _saved_object_references.extractSavedObjectReferences)(updatedSourceConfigurationAttributes);
    const updatedSourceConfiguration = convertSavedObjectToSavedSourceConfiguration(
    // update() will perform a deep merge. We use create() with overwrite: true instead. mergeSourceConfiguration()
    // ensures the correct and intended merging of properties.
    await savedObjectsClient.create(_saved_object_type.infraSourceConfigurationSavedObjectName, attributes, {
      id: sourceId,
      overwrite: true,
      references,
      version
    }));
    await this.metricsClient.updateMetricIndices({
      savedObjectsClient,
      metricIndices: updatedSourceConfiguration.configuration.metricAlias
    });
    return {
      ...updatedSourceConfiguration,
      configuration: mergeSourceConfiguration(_defaults.defaultSourceConfiguration, updatedSourceConfiguration.configuration)
    };
  }
  async getSavedSourceConfiguration(savedObjectsClient, sourceId) {
    const savedObject = await savedObjectsClient.get(_saved_object_type.infraSourceConfigurationSavedObjectName, sourceId);
    return convertSavedObjectToSavedSourceConfiguration(savedObject);
  }
}
exports.InfraSources = InfraSources;
const mergeSourceConfiguration = (first, ...others) => others.reduce((previousSourceConfiguration, currentSourceConfiguration) => ({
  ...previousSourceConfiguration,
  ...currentSourceConfiguration
}), first);
exports.mergeSourceConfiguration = mergeSourceConfiguration;
const convertSavedObjectToSavedSourceConfiguration = savedObject => (0, _pipeable.pipe)(_.SourceConfigurationSavedObjectRT.decode(savedObject), (0, _Either.map)(savedSourceConfiguration => ({
  id: savedSourceConfiguration.id,
  version: savedSourceConfiguration.version,
  updatedAt: savedSourceConfiguration.updated_at,
  origin: 'stored',
  configuration: (0, _saved_object_references.resolveSavedObjectReferences)(savedSourceConfiguration.attributes, savedObject.references)
})), (0, _Either.fold)(errors => {
  throw new Error((0, _PathReporter.failure)(errors).join('\n'));
}, _function.identity));
exports.convertSavedObjectToSavedSourceConfiguration = convertSavedObjectToSavedSourceConfiguration;