"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.correctCommonEsqlMistakes = correctCommonEsqlMistakes;
exports.splitIntoCommands = splitIntoCommands;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const STRING_DELIMITER_TOKENS = ['`', "'", '"'];
const ESCAPE_TOKEN = '\\\\';

// this function splits statements by a certain token,
// and takes into account escaping, or function calls

function split(value, splitToken) {
  const statements = [];
  let delimiterToken;
  let groupingCount = 0;
  let currentStatement = '';
  const trimmed = value.trim().split('');
  for (let index = 0; index < trimmed.length; index++) {
    const char = trimmed[index];
    if (!delimiterToken && groupingCount === 0 && trimmed.slice(index, index + splitToken.length).join('').toLowerCase() === splitToken.toLowerCase()) {
      index += splitToken.length - 1;
      statements.push(currentStatement.trim());
      currentStatement = '';
      continue;
    }
    currentStatement += char;
    if (delimiterToken === char) {
      // end identifier
      delimiterToken = undefined;
    } else if (!delimiterToken && trimmed[index - 1] !== ESCAPE_TOKEN) {
      const applicableToken = STRING_DELIMITER_TOKENS.includes(char) ? char : undefined;
      if (applicableToken) {
        // start identifier
        delimiterToken = applicableToken;
        continue;
      } else if (char === '(') {
        groupingCount++;
      } else if (char === ')') {
        groupingCount--;
      }
    }
  }
  if (currentStatement) {
    statements.push(currentStatement.trim());
  }
  return statements;
}
function splitIntoCommands(query) {
  const commands = split(query, '|');
  return commands.map(command => {
    var _command$match;
    const commandName = (_command$match = command.match(/^([A-Za-z]+)/)) === null || _command$match === void 0 ? void 0 : _command$match[1];
    return {
      name: commandName,
      command
    };
  });
}
function replaceSingleQuotesWithDoubleQuotes(command) {
  const regex = /'(?=(?:[^"]*"[^"]*")*[^"]*$)/g;
  return command.replace(regex, '"');
}
function removeColumnQuotesAndEscape(column) {
  const plainColumnIdentifier = column.replaceAll(/^"(.*)"$/g, `$1`).replaceAll(/^'(.*)'$/g, `$1`);
  if (isValidColumnName(plainColumnIdentifier)) {
    return plainColumnIdentifier;
  }
  return '`' + plainColumnIdentifier + '`';
}
function replaceAsKeywordWithAssignments(command) {
  return command.replaceAll(/^STATS\s*(.*)/g, (__, statsOperations) => {
    return `STATS ${statsOperations.replaceAll(/(,\s*)?(.*?)\s(AS|as)\s([`a-zA-Z0-9.\-_]+)/g, '$1$4 = $2')}`;
  });
}
function isValidColumnName(column) {
  return Boolean(column.match(/^`.*`$/) || column.match(/^[@A-Za-z\._\-\d]+$/));
}
function escapeColumns(line) {
  var _line$match;
  const [, command, body] = (_line$match = line.match(/^([A-Za-z_]+)(.*)$/ms)) !== null && _line$match !== void 0 ? _line$match : ['', '', ''];
  const escapedBody = split(body.trim(), ',').map(statement => {
    const [lhs, rhs] = split(statement, '=');
    if (!rhs) {
      return lhs;
    }
    return `${removeColumnQuotesAndEscape(lhs)} = ${rhs}`;
  }).join(', ');
  return `${command} ${escapedBody}`;
}
function verifyKeepColumns(keepCommand, nextCommands) {
  const columnsInKeep = split(keepCommand.replace(/^KEEP\s*/, ''), ',').map(statement => {
    var _split;
    return (_split = split(statement, '=')) === null || _split === void 0 ? void 0 : _split[0].trim();
  });
  const availableColumns = columnsInKeep.concat();
  for (const {
    name,
    command
  } of nextCommands) {
    if (['STATS', 'KEEP', 'DROP', 'DISSECT', 'GROK', 'ENRICH', 'RENAME'].includes(name || '')) {
      // these operations alter columns in a way that is hard to analyze, so we abort
      break;
    }
    const commandBody = command.replace(/^[A-Za-z]+\s*/, '');
    if (name === 'EVAL') {
      // EVAL creates new columns, make them available
      const columnsInEval = split(commandBody, ',').map(column => split(column.trim(), '=')[0].trim());
      columnsInEval.forEach(column => {
        availableColumns.push(column);
      });
    }
    if (name === 'RENAME') {
      // RENAME creates and removes columns
      split(commandBody, ',').forEach(statement => {
        const [prevName, newName] = split(statement, 'AS').map(side => side.trim());
        availableColumns.push(newName);
        if (!availableColumns.includes(prevName)) {
          columnsInKeep.push(prevName);
        }
      });
    }
    if (name === 'SORT') {
      const columnsInSort = split(commandBody, ',').map(column => split(column.trim(), ' ')[0].trim());
      columnsInSort.forEach(column => {
        if (isValidColumnName(column) && !availableColumns.includes(column)) {
          columnsInKeep.push(column);
        }
      });
    }
  }
  return `KEEP ${columnsInKeep.join(', ')}`;
}
function escapeExpressionsInSort(sortCommand) {
  const columnsInSort = split(sortCommand.replace(/^SORT\s*/, ''), ',').map(statement => {
    var _split2;
    return (_split2 = split(statement, '=')) === null || _split2 === void 0 ? void 0 : _split2[0].trim();
  }).map(columnAndSortOrder => {
    let [, column, sortOrder = ''] = columnAndSortOrder.match(/^(.*?)\s+(ASC|DESC\s*([A-Z\s]+)?)?$/i) || [];
    if (!column) {
      return columnAndSortOrder;
    }
    if (sortOrder) sortOrder = ` ${sortOrder}`;
    if (!column.match(/^`.*?`$/) && !column.match(/^[a-zA-Z0-9_\.@]+$/)) {
      column = `\`${column}\``;
    }
    return `${column}${sortOrder}`;
  });
  return `SORT ${columnsInSort.join(', ')}`;
}
function ensureEqualityOperators(whereCommand) {
  const body = whereCommand.split(/^WHERE /)[1];
  const byChar = body.split('');
  let next = '';
  let isColumnName = false;
  byChar.forEach((char, index) => {
    next += char;
    if (!isColumnName && char === '=' && byChar[index - 1] === ' ' && byChar[index + 1] === ' ') {
      next += '=';
    }
    if (!isColumnName && (char === '`' || char.match(/[a-z@]/i))) {
      isColumnName = true;
    } else if (isColumnName && (char === '`' || !char.match(/[a-z@0-9]/i))) {
      isColumnName = false;
    }
  });
  return `WHERE ${next}`;
}
function correctCommonEsqlMistakes(query) {
  const commands = splitIntoCommands(query.trim());
  const formattedCommands = commands.map(({
    name,
    command
  }, index) => {
    let formattedCommand = command;
    formattedCommand = formattedCommand.replaceAll(/"@timestamp"/g, '@timestamp').replaceAll(/'@timestamp'/g, '@timestamp');
    switch (name) {
      case 'FROM':
        {
          formattedCommand = split(formattedCommand, ',').map(singlePattern => singlePattern.replaceAll(/`/g, '"').replaceAll(/'/g, '"')).join(',');
          break;
        }
      case 'WHERE':
        formattedCommand = replaceSingleQuotesWithDoubleQuotes(formattedCommand);
        formattedCommand = ensureEqualityOperators(formattedCommand);
        break;
      case 'EVAL':
        formattedCommand = replaceSingleQuotesWithDoubleQuotes(formattedCommand);
        formattedCommand = escapeColumns(formattedCommand);
        break;
      case 'STATS':
        formattedCommand = replaceAsKeywordWithAssignments(formattedCommand);
        const [before, after] = split(formattedCommand, ' BY ');
        formattedCommand = escapeColumns(before);
        if (after) {
          formattedCommand += ` BY ${after}`;
        }
        break;
      case 'KEEP':
        formattedCommand = verifyKeepColumns(formattedCommand, commands.slice(index + 1));
        break;
      case 'SORT':
        formattedCommand = escapeExpressionsInSort(formattedCommand);
        break;
    }
    return formattedCommand;
  });
  const output = formattedCommands.join('\n| ');
  const originalFormattedQuery = commands.map(cmd => cmd.command).join('\n| ');
  const isCorrection = output !== originalFormattedQuery;
  return {
    input: query,
    output,
    isCorrection
  };
}