"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InferenceConnectorType = exports.COMPLETION_TASK_TYPE = void 0;
exports.isSupportedConnector = isSupportedConnector;
exports.isSupportedConnectorType = isSupportedConnectorType;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * The list of connector types that can be used with the inference APIs
 */
let InferenceConnectorType = exports.InferenceConnectorType = /*#__PURE__*/function (InferenceConnectorType) {
  InferenceConnectorType["OpenAI"] = ".gen-ai";
  InferenceConnectorType["Bedrock"] = ".bedrock";
  InferenceConnectorType["Gemini"] = ".gemini";
  InferenceConnectorType["Inference"] = ".inference";
  return InferenceConnectorType;
}({});
const COMPLETION_TASK_TYPE = exports.COMPLETION_TASK_TYPE = 'chat_completion';
const allSupportedConnectorTypes = Object.values(InferenceConnectorType);

/**
 * Represents a stack connector that can be used for inference.
 */

/**
 * Checks if a given connector type is compatible for inference.
 *
 * Note: this check is not sufficient to assert if a given connector can be
 * used for inference, as `.inference` connectors need additional check logic.
 * Please use `isSupportedConnector` instead when possible.
 */
function isSupportedConnectorType(id) {
  return allSupportedConnectorTypes.includes(id);
}

/**
 * Checks if a given connector is compatible for inference.
 *
 * A connector is compatible if:
 * 1. its type is in the list of allowed types
 * 2. for inference connectors, if its taskType is "chat_completion"
 */
function isSupportedConnector(connector) {
  if (!isSupportedConnectorType(connector.actionTypeId)) {
    return false;
  }
  if (connector.actionTypeId === InferenceConnectorType.Inference) {
    var _connector$config;
    const config = (_connector$config = connector.config) !== null && _connector$config !== void 0 ? _connector$config : {};
    // only chat_completion endpoint can be used for inference
    if (config.taskType !== COMPLETION_TASK_TYPE) {
      return false;
    }
  }
  return true;
}

/**
 * Connector types are living in the actions plugin and we can't afford
 * having dependencies from this package to some mid-level plugin,
 * so we're just using our own connector mixin type.
 */