"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateKafkaHost = exports.UpdateOutputSchema = exports.RemoteElasticSearchSchema = exports.OutputSchema = exports.OutputResponseSchema = exports.LogstashSchema = exports.KafkaSchema = exports.ElasticSearchSchema = void 0;
exports.validateLogstashHost = validateLogstashHost;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateLogstashHost(val) {
  if (val.match(/^http([s]){0,1}:\/\//)) {
    return 'Host address must begin with a domain name or IP address';
  }
  try {
    const url = new URL(`http://${val}`);
    if (url.host !== val.toLowerCase()) {
      return 'Invalid host';
    }
  } catch (err) {
    return 'Invalid Logstash host';
  }
}
const validateKafkaHost = input => {
  const parts = input.split(':');
  if (parts.length !== 2 || !parts[0] || parts[0].includes('://')) {
    return 'Invalid format. Expected "host:port" without protocol';
  }
  const port = parseInt(parts[1], 10);
  if (isNaN(port) || port < 1 || port > 65535) {
    return 'Invalid port number. Expected a number between 1 and 65535';
  }
  return undefined;
};
exports.validateKafkaHost = validateKafkaHost;
const secretRefSchema = _configSchema.schema.oneOf([_configSchema.schema.object({
  id: _configSchema.schema.string()
}), _configSchema.schema.string()]);

/**
 * Base schemas
 */

const BaseSchema = {
  id: _configSchema.schema.maybe(_configSchema.schema.string()),
  name: _configSchema.schema.string(),
  is_default: _configSchema.schema.boolean({
    defaultValue: false
  }),
  is_default_monitoring: _configSchema.schema.boolean({
    defaultValue: false
  }),
  is_internal: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  is_preconfigured: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  ca_sha256: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.string()])),
  ca_trusted_fingerprint: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.string()])),
  config_yaml: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.string()])),
  ssl: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.object({
    certificate_authorities: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
    certificate: _configSchema.schema.maybe(_configSchema.schema.string()),
    key: _configSchema.schema.maybe(_configSchema.schema.string()),
    verification_mode: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.kafkaVerificationModes.Full), _configSchema.schema.literal(_constants.kafkaVerificationModes.None), _configSchema.schema.literal(_constants.kafkaVerificationModes.Certificate), _configSchema.schema.literal(_constants.kafkaVerificationModes.Strict)]))
  })])),
  proxy_id: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.string()])),
  shipper: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.object({
    disk_queue_enabled: _configSchema.schema.nullable(_configSchema.schema.boolean({
      defaultValue: false
    })),
    disk_queue_path: _configSchema.schema.nullable(_configSchema.schema.string()),
    disk_queue_max_size: _configSchema.schema.nullable(_configSchema.schema.number()),
    disk_queue_encryption_enabled: _configSchema.schema.nullable(_configSchema.schema.boolean()),
    disk_queue_compression_enabled: _configSchema.schema.nullable(_configSchema.schema.boolean()),
    compression_level: _configSchema.schema.nullable(_configSchema.schema.number()),
    loadbalance: _configSchema.schema.nullable(_configSchema.schema.boolean()),
    mem_queue_events: _configSchema.schema.nullable(_configSchema.schema.number()),
    queue_flush_timeout: _configSchema.schema.nullable(_configSchema.schema.number()),
    max_batch_bytes: _configSchema.schema.nullable(_configSchema.schema.number())
  })])),
  allow_edit: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
};
const UpdateSchema = {
  ...BaseSchema,
  name: _configSchema.schema.maybe(_configSchema.schema.string()),
  is_default: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  is_default_monitoring: _configSchema.schema.maybe(_configSchema.schema.boolean())
};
const PresetSchema = _configSchema.schema.oneOf([_configSchema.schema.literal('balanced'), _configSchema.schema.literal('custom'), _configSchema.schema.literal('throughput'), _configSchema.schema.literal('scale'), _configSchema.schema.literal('latency')]);

/**
 * Elasticsearch schemas
 */

const ElasticSearchSchema = exports.ElasticSearchSchema = {
  ...BaseSchema,
  type: _configSchema.schema.literal(_constants.outputType.Elasticsearch),
  hosts: _configSchema.schema.arrayOf(_configSchema.schema.uri({
    scheme: ['http', 'https']
  }), {
    minSize: 1
  }),
  preset: _configSchema.schema.maybe(PresetSchema)
};
const ElasticSearchUpdateSchema = {
  ...UpdateSchema,
  type: _configSchema.schema.maybe(_configSchema.schema.literal(_constants.outputType.Elasticsearch)),
  hosts: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.uri({
    scheme: ['http', 'https']
  }), {
    minSize: 1
  })),
  preset: _configSchema.schema.maybe(PresetSchema)
};

/**
 * Remote Elasticsearch schemas
 */

const RemoteElasticSearchSchema = exports.RemoteElasticSearchSchema = {
  ...ElasticSearchSchema,
  type: _configSchema.schema.literal(_constants.outputType.RemoteElasticsearch),
  service_token: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.string()])),
  secrets: _configSchema.schema.maybe(_configSchema.schema.object({
    service_token: _configSchema.schema.maybe(secretRefSchema)
  }))
};
const RemoteElasticSearchUpdateSchema = {
  ...ElasticSearchUpdateSchema,
  type: _configSchema.schema.maybe(_configSchema.schema.literal(_constants.outputType.RemoteElasticsearch)),
  service_token: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.string()])),
  secrets: _configSchema.schema.maybe(_configSchema.schema.object({
    service_token: _configSchema.schema.maybe(secretRefSchema)
  }))
};

/**
 * Logstash schemas
 */

const LogstashSchema = exports.LogstashSchema = {
  ...BaseSchema,
  type: _configSchema.schema.literal(_constants.outputType.Logstash),
  hosts: _configSchema.schema.arrayOf(_configSchema.schema.string({
    validate: validateLogstashHost
  }), {
    minSize: 1
  }),
  secrets: _configSchema.schema.maybe(_configSchema.schema.object({
    ssl: _configSchema.schema.maybe(_configSchema.schema.object({
      key: _configSchema.schema.maybe(secretRefSchema)
    }))
  }))
};
const LogstashUpdateSchema = {
  ...UpdateSchema,
  type: _configSchema.schema.maybe(_configSchema.schema.literal(_constants.outputType.Logstash)),
  hosts: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string({
    validate: validateLogstashHost
  }), {
    minSize: 1
  })),
  secrets: _configSchema.schema.maybe(_configSchema.schema.object({
    ssl: _configSchema.schema.maybe(_configSchema.schema.object({
      key: _configSchema.schema.maybe(secretRefSchema)
    }))
  }))
};
const KafkaSchema = exports.KafkaSchema = {
  ...BaseSchema,
  type: _configSchema.schema.literal(_constants.outputType.Kafka),
  hosts: _configSchema.schema.arrayOf(_configSchema.schema.string({
    validate: validateKafkaHost
  }), {
    minSize: 1
  }),
  version: _configSchema.schema.maybe(_configSchema.schema.string()),
  key: _configSchema.schema.maybe(_configSchema.schema.string()),
  compression: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.kafkaCompressionType.Gzip), _configSchema.schema.literal(_constants.kafkaCompressionType.Snappy), _configSchema.schema.literal(_constants.kafkaCompressionType.Lz4), _configSchema.schema.literal(_constants.kafkaCompressionType.None)])),
  compression_level: _configSchema.schema.conditional(_configSchema.schema.siblingRef('compression'), _configSchema.schema.string({
    validate: val => val === _constants.kafkaCompressionType.Gzip ? undefined : 'never'
  }), _configSchema.schema.number(), _configSchema.schema.never()),
  client_id: _configSchema.schema.maybe(_configSchema.schema.string()),
  auth_type: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.kafkaAuthType.None), _configSchema.schema.literal(_constants.kafkaAuthType.Userpass), _configSchema.schema.literal(_constants.kafkaAuthType.Ssl), _configSchema.schema.literal(_constants.kafkaAuthType.Kerberos)]),
  connection_type: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth_type'), _constants.kafkaAuthType.None, _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.kafkaConnectionType.Plaintext), _configSchema.schema.literal(_constants.kafkaConnectionType.Encryption)]), _configSchema.schema.never()),
  username: _configSchema.schema.nullable(_configSchema.schema.conditional(_configSchema.schema.siblingRef('auth_type'), _constants.kafkaAuthType.Userpass, _configSchema.schema.string(), _configSchema.schema.never())),
  password: _configSchema.schema.nullable(_configSchema.schema.conditional(_configSchema.schema.siblingRef('secrets.password'), secretRefSchema, _configSchema.schema.never(), _configSchema.schema.conditional(_configSchema.schema.siblingRef('username'), _configSchema.schema.string(), _configSchema.schema.string(), _configSchema.schema.never()))),
  sasl: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(null), _configSchema.schema.object({
    mechanism: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.kafkaSaslMechanism.Plain), _configSchema.schema.literal(_constants.kafkaSaslMechanism.ScramSha256), _configSchema.schema.literal(_constants.kafkaSaslMechanism.ScramSha512)]))
  })])),
  partition: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.kafkaPartitionType.Random), _configSchema.schema.literal(_constants.kafkaPartitionType.RoundRobin), _configSchema.schema.literal(_constants.kafkaPartitionType.Hash)])),
  random: _configSchema.schema.maybe(_configSchema.schema.object({
    group_events: _configSchema.schema.maybe(_configSchema.schema.number())
  })),
  round_robin: _configSchema.schema.maybe(_configSchema.schema.object({
    group_events: _configSchema.schema.maybe(_configSchema.schema.number())
  })),
  hash: _configSchema.schema.maybe(_configSchema.schema.object({
    hash: _configSchema.schema.maybe(_configSchema.schema.string()),
    random: _configSchema.schema.maybe(_configSchema.schema.boolean())
  })),
  topic: _configSchema.schema.maybe(_configSchema.schema.string()),
  headers: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
    key: _configSchema.schema.string(),
    value: _configSchema.schema.string()
  }))),
  timeout: _configSchema.schema.maybe(_configSchema.schema.number()),
  broker_timeout: _configSchema.schema.maybe(_configSchema.schema.number()),
  required_acks: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(1), _configSchema.schema.literal(0), _configSchema.schema.literal(-1)])),
  secrets: _configSchema.schema.maybe(_configSchema.schema.object({
    password: _configSchema.schema.maybe(secretRefSchema),
    ssl: _configSchema.schema.maybe(_configSchema.schema.object({
      key: secretRefSchema
    }))
  }))
};
const KafkaUpdateSchema = {
  ...UpdateSchema,
  ...KafkaSchema,
  type: _configSchema.schema.maybe(_configSchema.schema.literal(_constants.outputType.Kafka)),
  hosts: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string({
    validate: validateKafkaHost
  }), {
    minSize: 1
  })),
  auth_type: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.kafkaAuthType.None), _configSchema.schema.literal(_constants.kafkaAuthType.Userpass), _configSchema.schema.literal(_constants.kafkaAuthType.Ssl), _configSchema.schema.literal(_constants.kafkaAuthType.Kerberos)]))
};
const OutputSchema = exports.OutputSchema = _configSchema.schema.oneOf([_configSchema.schema.object({
  ...ElasticSearchSchema
}), _configSchema.schema.object({
  ...RemoteElasticSearchSchema
}), _configSchema.schema.object({
  ...LogstashSchema
}), _configSchema.schema.object({
  ...KafkaSchema
})]);
const OutputResponseSchema = exports.OutputResponseSchema = _configSchema.schema.object({
  item: OutputSchema.extendsDeep({
    unknowns: 'allow'
  })
});
const UpdateOutputSchema = exports.UpdateOutputSchema = _configSchema.schema.oneOf([_configSchema.schema.object({
  ...ElasticSearchUpdateSchema
}), _configSchema.schema.object({
  ...RemoteElasticSearchUpdateSchema
}), _configSchema.schema.object({
  ...LogstashUpdateSchema
}), _configSchema.schema.object({
  ...KafkaUpdateSchema
})]);