"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cleanupComponentTemplate = cleanupComponentTemplate;
exports.cleanupTransforms = cleanupTransforms;
exports.deleteESAssets = exports.deleteESAsset = void 0;
exports.deleteILMPolicies = deleteILMPolicies;
exports.deleteKibanaAssets = deleteKibanaAssets;
exports.deleteKibanaSavedObjectsAssets = deleteKibanaSavedObjectsAssets;
exports.deleteMLModels = deleteMLModels;
exports.deletePrerequisiteAssets = deletePrerequisiteAssets;
exports.removeInstallation = removeInstallation;
exports.splitESAssets = void 0;
var _server = require("@kbn/core/server");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _minVersion = _interopRequireDefault(require("semver/ranges/min-version"));
var _lodash = require("lodash");
var _pMap = _interopRequireDefault(require("p-map"));
var _update_settings = require("../elasticsearch/index/update_settings");
var _constants2 = require("../../../constants");
var _types = require("../../../types");
var _ingest_pipeline = require("../elasticsearch/ingest_pipeline");
var _install = require("../kibana/index_pattern/install");
var _remove = require("../elasticsearch/transform/remove");
var _ml_model = require("../elasticsearch/ml_model");
var _ = require("../..");
var _archive = require("../archive");
var _remove2 = require("../elasticsearch/datastream_ilm/remove");
var _storage = require("../archive/storage");
var _audit_logging = require("../../audit_logging");
var _errors = require("../../../errors");
var _populate_package_policy_assigned_agents_count = require("../../package_policies/populate_package_policy_assigned_agents_count");
var Registry = _interopRequireWildcard(require("../registry"));
var _2 = require(".");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_ASSETS_TO_DELETE = 1000;
async function removeInstallation(options) {
  const {
    savedObjectsClient,
    pkgName,
    esClient
  } = options;
  const installation = await (0, _2.getInstallation)({
    savedObjectsClient,
    pkgName
  });
  if (!installation) {
    throw new _errors.PackageRemovalError(`${pkgName} is not installed`);
  }
  const {
    total,
    items
  } = await _.packagePolicyService.list(_.appContextService.getInternalUserSOClientWithoutSpaceExtension(), {
    kuery: `${_constants2.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${pkgName}`,
    page: 1,
    perPage: _constants2.SO_SEARCH_LIMIT,
    spaceId: '*'
  });
  if (!options.force) {
    await (0, _populate_package_policy_assigned_agents_count.populatePackagePolicyAssignedAgentsCount)(esClient, items);
  }
  if (total > 0) {
    if (options.force || items.every(item => {
      var _item$agents;
      return ((_item$agents = item.agents) !== null && _item$agents !== void 0 ? _item$agents : 0) === 0;
    })) {
      // delete package policies
      const ids = items.map(item => item.id);
      await _.packagePolicyService.delete(savedObjectsClient, esClient, ids, {
        force: options.force
      });
    } else {
      throw new _errors.PackageRemovalError(`Unable to remove package with existing package policy(s) in use by agent(s)`);
    }
  }

  // Delete the installed assets. Don't include installation.package_assets. Those are irrelevant to users
  const installedAssets = [...installation.installed_kibana, ...installation.installed_es];
  await deleteAssets(installation, esClient);

  // Delete the manager saved object with references to the asset objects
  // could also update with [] or some other state
  _audit_logging.auditLoggingService.writeCustomSoAuditLog({
    action: 'delete',
    id: pkgName,
    savedObjectType: _constants2.PACKAGES_SAVED_OBJECT_TYPE
  });
  await savedObjectsClient.delete(_constants2.PACKAGES_SAVED_OBJECT_TYPE, pkgName);

  // delete the index patterns if no packages are installed
  // this must be done after deleting the saved object for the current package otherwise it will retrieve the package
  // from the registry again and keep the index patterns
  await (0, _install.removeUnusedIndexPatterns)(savedObjectsClient);

  // remove the package archive and its contents from the cache so that a reinstall fetches
  // a fresh copy from the registry
  (0, _archive.deletePackageCache)({
    name: pkgName,
    version: installation.version
  });
  await (0, _storage.removeArchiveEntries)({
    savedObjectsClient,
    refs: installation.package_assets
  });

  // successful delete's in SO client return {}. return something more useful
  return installedAssets;
}

/**
 * This method deletes saved objects resolving them whenever necessary.
 *
 * Resolving is needed when deleting assets that were installed in 7.x to
 * mitigate the breaking change that occurred in 8.0. This is a memory-intensive
 * operation as it requires loading all the saved objects into memory. It is
 * generally better to delete assets directly if the package is known to be
 * installed in 8.x or later.
 */
async function deleteKibanaAssets({
  installedObjects,
  packageInfo,
  spaceId = _constants.DEFAULT_SPACE_ID
}) {
  var _packageInfo$conditio, _packageInfo$conditio2;
  const savedObjectsClient = new _server.SavedObjectsClient(_.appContextService.getSavedObjects().createInternalRepository());
  const namespace = _server.SavedObjectsUtils.namespaceStringToId(spaceId);

  // TODO this should be the installed package info, not the package that is being installed
  const minKibana = (_packageInfo$conditio = packageInfo.conditions) !== null && _packageInfo$conditio !== void 0 && (_packageInfo$conditio2 = _packageInfo$conditio.kibana) !== null && _packageInfo$conditio2 !== void 0 && _packageInfo$conditio2.version ? (0, _minVersion.default)(packageInfo.conditions.kibana.version) : null;

  // Compare Kibana versions to determine if the package could been installed
  // only in 8.x or later. If so, we can skip SO resolution step altogether
  // and delete the assets directly. Otherwise, we need to resolve the assets
  // which might create high memory pressure if a package has a lot of assets.
  if (minKibana && minKibana.major >= 8) {
    await bulkDeleteSavedObjects(installedObjects, namespace, savedObjectsClient);
  } else {
    const {
      resolved_objects: resolvedObjects
    } = await savedObjectsClient.bulkResolve(installedObjects, {
      namespace
    });
    for (const {
      saved_object: savedObject
    } of resolvedObjects) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: savedObject.id,
        savedObjectType: savedObject.type
      });
    }
    const foundObjects = resolvedObjects.filter(({
      saved_object: savedObject
    }) => {
      var _savedObject$error;
      return (savedObject === null || savedObject === void 0 ? void 0 : (_savedObject$error = savedObject.error) === null || _savedObject$error === void 0 ? void 0 : _savedObject$error.statusCode) !== 404;
    });

    // in the case of a partial install, it is expected that some assets will be not found
    // we filter these out before calling delete
    const assetsToDelete = foundObjects.map(({
      saved_object: {
        id,
        type
      }
    }) => ({
      id,
      type
    }));
    await bulkDeleteSavedObjects(assetsToDelete, namespace, savedObjectsClient);
  }
}
async function bulkDeleteSavedObjects(assetsToDelete, namespace, savedObjectsClient) {
  for (const asset of assetsToDelete) {
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'delete',
      id: asset.id,
      savedObjectType: asset.type
    });
  }

  // Delete assets in chunks to avoid high memory pressure. This is mostly
  // relevant for packages containing many assets, as large payload and response
  // objects are created in memory during the delete operation. While chunking
  // may work slower, it allows garbage collection to clean up memory between
  // requests.
  for (const assetsChunk of (0, _lodash.chunk)(assetsToDelete, MAX_ASSETS_TO_DELETE)) {
    await savedObjectsClient.bulkDelete(assetsChunk, {
      namespace
    });
  }
}
const deleteESAsset = async (installedObject, esClient) => {
  const {
    id,
    type
  } = installedObject;
  const assetType = type;
  if (assetType === _types.ElasticsearchAssetType.ingestPipeline) {
    return (0, _ingest_pipeline.deletePipeline)(esClient, id);
  } else if (assetType === _types.ElasticsearchAssetType.indexTemplate) {
    return deleteIndexTemplate(esClient, id);
  } else if (assetType === _types.ElasticsearchAssetType.componentTemplate) {
    return deleteComponentTemplate(esClient, id);
  } else if (assetType === _types.ElasticsearchAssetType.transform) {
    return (0, _remove.deleteTransforms)(esClient, [id], true);
  } else if (assetType === _types.ElasticsearchAssetType.dataStreamIlmPolicy) {
    return (0, _remove2.deleteIlms)(esClient, [id]);
  } else if (assetType === _types.ElasticsearchAssetType.ilmPolicy) {
    return (0, _remove2.deleteIlms)(esClient, [id]);
  } else if (assetType === _types.ElasticsearchAssetType.mlModel) {
    return (0, _ml_model.deleteMlModel)(esClient, [id]);
  }
};
exports.deleteESAsset = deleteESAsset;
const deleteESAssets = (installedObjects, esClient) => {
  return installedObjects.map(installedObject => deleteESAsset(installedObject, esClient));
};
exports.deleteESAssets = deleteESAssets;
const splitESAssets = installedEs => {
  const [indexTemplatesAndPipelines, indexAssets, transformAssets, otherAssets] = installedEs.reduce(([indexTemplateAndPipelineTypes, indexAssetTypes, transformAssetTypes, otherAssetTypes], asset) => {
    if (asset.type === _types.ElasticsearchAssetType.indexTemplate || asset.type === _types.ElasticsearchAssetType.ingestPipeline) {
      indexTemplateAndPipelineTypes.push(asset);
    } else if (asset.type === _types.ElasticsearchAssetType.index) {
      indexAssetTypes.push(asset);
    } else if (asset.type === _types.ElasticsearchAssetType.transform) {
      transformAssetTypes.push(asset);
    } else {
      otherAssetTypes.push(asset);
    }
    return [indexTemplateAndPipelineTypes, indexAssetTypes, transformAssetTypes, otherAssetTypes];
  }, [[], [], [], []]);
  return {
    indexTemplatesAndPipelines,
    indexAssets,
    transformAssets,
    otherAssets
  };
};

/**
 * deletePrerequisiteAssets removes the ES assets that need to be deleted first and in a certain order.
 * All the other assets can be deleted after these (see deleteAssets)
 */
exports.splitESAssets = splitESAssets;
async function deletePrerequisiteAssets({
  indexAssets,
  transformAssets,
  indexTemplatesAndPipelines
}, esClient) {
  const logger = _.appContextService.getLogger();
  // must unset default_pipelines settings in indices first, or pipelines associated with an index cannot not be deleted
  // must delete index templates first, or component templates which reference them cannot be deleted
  // must delete ingestPipelines first, or ml models referenced in them cannot be deleted.
  // separate the assets into Index Templates and other assets.

  try {
    // must first unset any default pipeline associated with any existing indices
    // by setting empty string
    await (0, _pMap.default)(indexAssets, asset => (0, _update_settings.updateIndexSettings)(esClient, asset.id, {
      default_pipeline: ''
    }), {
      concurrency: _constants2.MAX_CONCURRENT_ES_ASSETS_OPERATIONS
    });

    // in case transform's destination index contains any pipeline,
    // we should delete the transforms first
    await (0, _pMap.default)(transformAssets, transformAsset => deleteESAsset(transformAsset, esClient), {
      concurrency: _constants2.MAX_CONCURRENT_ES_ASSETS_OPERATIONS
    });

    // then delete index templates and pipelines
    await (0, _pMap.default)(indexTemplatesAndPipelines, asset => deleteESAsset(asset, esClient), {
      concurrency: _constants2.MAX_CONCURRENT_ES_ASSETS_OPERATIONS
    });
  } catch (err) {
    // in the rollback case, partial installs are likely, so missing assets are not an error
    if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      logger.error(err);
    }
  }
}
async function deleteAssets({
  installed_es: installedEs,
  installed_kibana: installedKibana,
  installed_kibana_space_id: spaceId = _constants.DEFAULT_SPACE_ID,
  additional_spaces_installed_kibana: installedInAdditionalSpacesKibana = {},
  name,
  version
}, esClient) {
  const logger = _.appContextService.getLogger();
  const {
    indexTemplatesAndPipelines,
    indexAssets,
    transformAssets,
    otherAssets
  } = splitESAssets(installedEs);

  // delete assets that need to be deleted first
  await deletePrerequisiteAssets({
    indexAssets,
    transformAssets,
    indexTemplatesAndPipelines
  }, esClient);

  // delete the other asset types
  try {
    const packageInfo = await Registry.fetchInfo(name, version);
    await Promise.all([...deleteESAssets(otherAssets, esClient), deleteKibanaAssets({
      installedObjects: installedKibana,
      spaceId,
      packageInfo
    }), Object.entries(installedInAdditionalSpacesKibana).map(([additionalSpaceId, kibanaAssets]) => deleteKibanaAssets({
      installedObjects: kibanaAssets,
      spaceId: additionalSpaceId,
      packageInfo
    }))]);
  } catch (err) {
    // in the rollback case, partial installs are likely, so missing assets are not an error
    if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      logger.error(err);
    }
  }
}
async function deleteIndexTemplate(esClient, name) {
  // '*' shouldn't ever appear here, but it still would delete all templates
  if (name && name !== '*') {
    try {
      await esClient.indices.deleteIndexTemplate({
        name
      }, {
        ignore: [404]
      });
    } catch {
      throw new _errors.FleetError(`Error deleting index template ${name}`);
    }
  }
}
async function deleteComponentTemplate(esClient, name) {
  // '*' shouldn't ever appear here, but it still would delete all templates
  if (name && name !== '*' && !name.endsWith(_constants2.USER_SETTINGS_TEMPLATE_SUFFIX)) {
    try {
      await esClient.cluster.deleteComponentTemplate({
        name
      }, {
        ignore: [404]
      });
    } catch (error) {
      throw new _errors.FleetError(`Error deleting component template ${name}`);
    }
  }
}
async function deleteKibanaSavedObjectsAssets({
  installedPkg,
  spaceId
}) {
  const {
    installed_kibana_space_id: installedSpaceId
  } = installedPkg.attributes;
  let refsToDelete;
  let spaceIdToDelete;
  if (!spaceId || spaceId === installedSpaceId) {
    refsToDelete = installedPkg.attributes.installed_kibana;
    spaceIdToDelete = installedSpaceId;
  } else {
    var _installedPkg$attribu, _installedPkg$attribu2;
    refsToDelete = (_installedPkg$attribu = (_installedPkg$attribu2 = installedPkg.attributes.additional_spaces_installed_kibana) === null || _installedPkg$attribu2 === void 0 ? void 0 : _installedPkg$attribu2[spaceId]) !== null && _installedPkg$attribu !== void 0 ? _installedPkg$attribu : [];
    spaceIdToDelete = spaceId;
  }
  if (!refsToDelete.length) return;
  const logger = _.appContextService.getLogger();
  const assetsToDelete = refsToDelete.filter(({
    type
  }) => _2.kibanaSavedObjectTypes.includes(type)).map(({
    id,
    type
  }) => ({
    id,
    type
  }));
  try {
    const packageInfo = await Registry.fetchInfo(installedPkg.attributes.name, installedPkg.attributes.version);
    await deleteKibanaAssets({
      installedObjects: assetsToDelete,
      spaceId: spaceIdToDelete,
      packageInfo
    });
  } catch (err) {
    // in the rollback case, partial installs are likely, so missing assets are not an error
    if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      logger.error(err);
    }
  }
}
function deleteILMPolicies(installedObjects, esClient) {
  const idsToDelete = installedObjects.filter(asset => asset.type === _types.ElasticsearchAssetType.dataStreamIlmPolicy || asset.type === _types.ElasticsearchAssetType.ilmPolicy).map(asset => asset.id);
  return (0, _remove2.deleteIlms)(esClient, idsToDelete);
}
function deleteMLModels(installedObjects, esClient) {
  const idsToDelete = installedObjects.filter(asset => asset.type === _types.ElasticsearchAssetType.mlModel).map(asset => asset.id);
  return (0, _ml_model.deleteMlModel)(esClient, idsToDelete);
}
function cleanupComponentTemplate(installedObjects, esClient) {
  const idsToDelete = installedObjects.filter(asset => asset.type === _types.ElasticsearchAssetType.mlModel).map(asset => asset.id);
  return deleteComponentTemplate(esClient, idsToDelete[0]);
}
function cleanupTransforms(installedObjects, esClient) {
  const idsToDelete = installedObjects.filter(asset => asset.type === _types.ElasticsearchAssetType.transform).map(asset => asset.id);
  return (0, _remove.deleteTransforms)(esClient, idsToDelete);
}