"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPackageSpecTagId = void 0;
exports.tagKibanaAssets = tagKibanaAssets;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _pMap = _interopRequireDefault(require("p-map"));
var _constants = require("@kbn/saved-objects-tagging-plugin/common/constants");
var _constants2 = require("../../../../constants");
var _app_context = require("../../../app_context");
var _install = require("./install");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MANAGED_TAG_COLOR = '#0077CC';
const PACKAGE_TAG_COLOR = '#4DD2CA';
const MANAGED_TAG_NAME = 'Managed';
const LEGACY_MANAGED_TAG_ID = 'managed';
const SECURITY_SOLUTION_TAG_NAME = 'Security Solution';
const SECURITY_SOLUTION_TAG_ID_BASE = 'security-solution';

// the tag service only accepts 6-digits hex colors
const TAG_COLORS = ['#FEC514', '#F583B7', '#F04E98', '#00BFB3', '#FEC514', '#BADA55', '#FFA500', '#9696F1', '#D36086', '#54B399', '#AAA8A5', '#A0A0A0'];
const getManagedTagId = spaceId => `fleet-managed-${spaceId}`;
const getPackageTagId = (spaceId, pkgName) => `fleet-pkg-${pkgName}-${spaceId}`;
const getLegacyPackageTagId = pkgName => pkgName;

/*
  This function is exported via fleet/plugin.ts to make it available to other plugins
  The `SecuritySolution` tag is a special case that needs to be handled separately
  In that case return id `security-solution-default`
*/
const getPackageSpecTagId = (spaceId, pkgName, tagName) => {
  if (tagName.toLowerCase() === SECURITY_SOLUTION_TAG_NAME.toLowerCase()) {
    return `${SECURITY_SOLUTION_TAG_ID_BASE}-${spaceId}`;
  }

  // UUID v5 needs a namespace (uuid.DNS) to generate a predictable uuid
  const uniqueId = (0, _uuid.v5)(`${tagName.toLowerCase()}`, _uuid.v5.DNS);
  return `fleet-shared-tag-${pkgName}-${uniqueId}-${spaceId}`;
};
exports.getPackageSpecTagId = getPackageSpecTagId;
const getRandomColor = () => {
  const randomizedIndex = Math.floor(Math.random() * TAG_COLORS.length);
  return TAG_COLORS[randomizedIndex];
};
async function tagKibanaAssets(opts) {
  const {
    savedObjectTagAssignmentService,
    kibanaAssets,
    importedAssets
  } = opts;
  const getNewId = assetId => {
    var _importedAssets$find$, _importedAssets$find;
    return (_importedAssets$find$ = (_importedAssets$find = importedAssets.find(imported => imported.id === assetId)) === null || _importedAssets$find === void 0 ? void 0 : _importedAssets$find.destinationId) !== null && _importedAssets$find$ !== void 0 ? _importedAssets$find$ : assetId;
  };
  const taggableAssets = getTaggableAssets(kibanaAssets).map(asset => ({
    ...asset,
    id: getNewId(asset.id)
  }));
  if (taggableAssets.length > 0) {
    const [managedTagId, packageTagId] = await Promise.all([ensureManagedTag(opts), ensurePackageTag(opts)]);
    try {
      await savedObjectTagAssignmentService.updateTagAssignments({
        tags: [managedTagId, packageTagId],
        assign: taggableAssets,
        unassign: [],
        refresh: false
      });
    } catch (error) {
      if (error.status === 404) {
        _app_context.appContextService.getLogger().warn(error.message);
        return;
      }
      throw error;
    }
    const packageSpecAssets = await getPackageSpecTags(taggableAssets, opts);
    const groupedAssets = groupByAssetId(packageSpecAssets);
    if (Object.entries(groupedAssets).length > 0) {
      await (0, _pMap.default)(Object.entries(groupedAssets), async ([assetId, asset]) => {
        try {
          await savedObjectTagAssignmentService.updateTagAssignments({
            tags: asset.tags,
            assign: [{
              id: assetId,
              type: asset.type
            }],
            unassign: [],
            refresh: false
          });
        } catch (error) {
          if (error.status === 404) {
            _app_context.appContextService.getLogger().warn(error.message);
            return;
          }
          throw error;
        }
      }, {
        concurrency: _constants2.MAX_CONCURRENT_PACKAGE_ASSETS
      });
    }
  }
}
function getTaggableAssets(kibanaAssets) {
  return Object.entries(kibanaAssets).flatMap(([assetType, assets]) => {
    if (!_constants.taggableTypes.includes(_install.KibanaSavedObjectTypeMapping[assetType])) {
      return [];
    }
    if (!assets.length) {
      return [];
    }
    return assets;
  });
}
async function ensureManagedTag(opts) {
  const {
    spaceId,
    savedObjectTagClient
  } = opts;
  const managedTagId = getManagedTagId(spaceId);
  const managedTag = await savedObjectTagClient.get(managedTagId).catch(() => {});
  if (managedTag) return managedTagId;
  const legacyManagedTag = await savedObjectTagClient.get(LEGACY_MANAGED_TAG_ID).catch(() => {});
  if (legacyManagedTag) return LEGACY_MANAGED_TAG_ID;
  await savedObjectTagClient.create({
    name: MANAGED_TAG_NAME,
    description: '',
    color: MANAGED_TAG_COLOR
  }, {
    id: managedTagId,
    overwrite: true,
    refresh: false,
    managed: true
  });
  return managedTagId;
}
async function ensurePackageTag(opts) {
  const {
    spaceId,
    savedObjectTagClient,
    pkgName,
    pkgTitle
  } = opts;
  const packageTagId = getPackageTagId(spaceId, pkgName);
  const packageTag = await savedObjectTagClient.get(packageTagId).catch(() => {});
  if (packageTag) return packageTagId;
  const legacyPackageTagId = getLegacyPackageTagId(pkgName);
  const legacyPackageTag = await savedObjectTagClient.get(legacyPackageTagId).catch(() => {});
  if (legacyPackageTag) return legacyPackageTagId;
  await savedObjectTagClient.create({
    name: pkgTitle,
    description: '',
    color: PACKAGE_TAG_COLOR
  }, {
    id: packageTagId,
    overwrite: true,
    refresh: false,
    managed: true
  });
  return packageTagId;
}

// Ensure that asset tags coming from the kibana/tags.yml file are correctly parsed and created
async function getPackageSpecTags(taggableAssets, opts) {
  const {
    spaceId,
    savedObjectTagClient,
    pkgName,
    assetTags
  } = opts;
  if (!assetTags || (assetTags === null || assetTags === void 0 ? void 0 : assetTags.length) === 0) return [];
  const assetsWithTags = await Promise.all(assetTags.map(async tag => {
    const uniqueTagId = getPackageSpecTagId(spaceId, pkgName, tag.text);
    const existingPackageSpecTag = await savedObjectTagClient.get(uniqueTagId).catch(() => {});
    if (!existingPackageSpecTag) {
      await savedObjectTagClient.create({
        name: tag.text,
        description: 'Tag defined in package-spec',
        color: getRandomColor()
      }, {
        id: uniqueTagId,
        overwrite: true,
        refresh: false,
        managed: true
      });
    }
    const assetTypes = getAssetTypesObjectReferences(tag === null || tag === void 0 ? void 0 : tag.asset_types, taggableAssets);
    const assetIds = getAssetIdsObjectReferences(tag === null || tag === void 0 ? void 0 : tag.asset_ids, taggableAssets);
    const totAssetsToAssign = assetTypes.concat(assetIds);
    const assetsToAssign = totAssetsToAssign.length > 0 ? (0, _lodash.uniqBy)(totAssetsToAssign, 'id') : [];
    return {
      tagId: uniqueTagId,
      assets: assetsToAssign
    };
  }));
  return assetsWithTags;
}

// Get all the assets of types defined in tag.asset_types from taggable kibanaAssets
const getAssetTypesObjectReferences = (assetTypes, taggableAssets) => {
  if (!assetTypes || assetTypes.length === 0) return [];
  return taggableAssets.filter(taggable => assetTypes.includes(taggable.type)).map(assetType => {
    return {
      type: assetType.type,
      id: assetType.id
    };
  });
};

// Get the references to ids defined in tag.asset_ids from taggable kibanaAssets
const getAssetIdsObjectReferences = (assetIds, taggableAssets) => {
  if (!assetIds || assetIds.length === 0) return [];
  return taggableAssets.filter(taggable => assetIds.includes(taggable.id)).map(assetType => {
    return {
      type: assetType.type,
      id: assetType.id
    };
  });
};

// Utility function that groups the assets by asset id
// It makes easier to update the tags in batches
const groupByAssetId = packageSpecsAssets => {
  if (packageSpecsAssets.length === 0) return {};
  const groupedAssets = {};
  packageSpecsAssets.forEach(({
    tagId,
    assets
  }) => {
    assets.forEach(asset => {
      const {
        id
      } = asset;
      if (!groupedAssets[id]) {
        groupedAssets[id] = {
          type: asset.type,
          tags: []
        };
      }
      groupedAssets[id].tags.push(tagId);
    });
  });
  return groupedAssets;
};