"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.reassignAgent = reassignAgent;
exports.reassignAgents = reassignAgents;
var _common = require("@kbn/kibana-utils-plugin/common");
var _agent_policy = require("../agent_policy");
var _errors = require("../../errors");
var _constants = require("../../constants");
var _agent_namespaces = require("../spaces/agent_namespaces");
var _get_current_namespace = require("../spaces/get_current_namespace");
var _crud = require("./crud");
var _actions = require("./actions");
var _reassign_action_runner = require("./reassign_action_runner");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function verifyNewAgentPolicy(soClient, newAgentPolicyId) {
  var _newAgentPolicy;
  let newAgentPolicy;
  try {
    newAgentPolicy = await _agent_policy.agentPolicyService.get(soClient, newAgentPolicyId);
  } catch (err) {
    if (err instanceof _common.SavedObjectNotFound) {
      throw new _errors.AgentPolicyNotFoundError(`Agent policy not found: ${newAgentPolicyId}`);
    }
  }
  if (!newAgentPolicy) {
    throw new _errors.AgentPolicyNotFoundError(`Agent policy not found: ${newAgentPolicyId}`);
  }
  if ((_newAgentPolicy = newAgentPolicy) !== null && _newAgentPolicy !== void 0 && _newAgentPolicy.is_managed) {
    throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot reassign agents to hosted agent policy ${newAgentPolicy.id}`);
  }
}
async function reassignAgent(soClient, esClient, agentId, newAgentPolicyId) {
  await verifyNewAgentPolicy(soClient, newAgentPolicyId);
  await (0, _crud.getAgentById)(esClient, soClient, agentId); // throw 404 if agent not in namespace

  const agentPolicy = await (0, _crud.getAgentPolicyForAgent)(soClient, esClient, agentId);
  if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_managed) {
    throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot reassign an agent from hosted agent policy ${agentPolicy.id}`);
  }
  const newAgentPolicy = await _agent_policy.agentPolicyService.get(soClient, newAgentPolicyId);
  await (0, _crud.updateAgent)(esClient, agentId, {
    policy_id: newAgentPolicyId,
    policy_revision: null,
    ...(newAgentPolicy !== null && newAgentPolicy !== void 0 && newAgentPolicy.space_ids ? {
      namespaces: newAgentPolicy.space_ids
    } : {})
  });
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  await (0, _actions.createAgentAction)(esClient, {
    agents: [agentId],
    created_at: new Date().toISOString(),
    type: 'POLICY_REASSIGN',
    data: {
      policy_id: newAgentPolicyId
    },
    namespaces: [currentSpaceId]
  });
}
async function reassignAgents(soClient, esClient, options, newAgentPolicyId) {
  await verifyNewAgentPolicy(soClient, newAgentPolicyId);
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  const outgoingErrors = {};
  let givenAgents = [];
  if ('agents' in options) {
    givenAgents = options.agents;
  } else if ('agentIds' in options) {
    const maybeAgents = await (0, _crud.getAgentsById)(esClient, soClient, options.agentIds);
    for (const maybeAgent of maybeAgents) {
      if ('notFound' in maybeAgent) {
        outgoingErrors[maybeAgent.id] = new _errors.AgentReassignmentError(`Cannot find agent ${maybeAgent.id}`);
      } else {
        givenAgents.push(maybeAgent);
      }
    }
  } else if ('kuery' in options) {
    var _options$batchSize, _options$showInactive;
    const batchSize = (_options$batchSize = options.batchSize) !== null && _options$batchSize !== void 0 ? _options$batchSize : _constants.SO_SEARCH_LIMIT;
    const namespaceFilter = await (0, _agent_namespaces.agentsKueryNamespaceFilter)(currentSpaceId);
    const kuery = namespaceFilter ? `${namespaceFilter} AND ${options.kuery}` : options.kuery;
    const res = await (0, _crud.getAgentsByKuery)(esClient, soClient, {
      kuery,
      showInactive: (_options$showInactive = options.showInactive) !== null && _options$showInactive !== void 0 ? _options$showInactive : false,
      page: 1,
      perPage: batchSize
    });
    // running action in async mode for >10k agents (or actions > batchSize for testing purposes)
    if (res.total <= batchSize) {
      givenAgents = res.agents;
    } else {
      return await new _reassign_action_runner.ReassignActionRunner(esClient, soClient, {
        ...options,
        spaceId: currentSpaceId,
        batchSize,
        total: res.total,
        newAgentPolicyId
      }, {
        pitId: await (0, _crud.openPointInTime)(esClient)
      }).runActionAsyncTask();
    }
  }
  return await (0, _reassign_action_runner.reassignBatch)(esClient, {
    newAgentPolicyId,
    spaceId: currentSpaceId
  }, givenAgents, outgoingErrors);
}