"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = void 0;
var _constants = require("../../../common/constants");
var _api_privileges = require("../../constants/api_privileges");
var _constants2 = require("../../constants");
var _types = require("../../types");
var _errors = require("../schema/errors");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerRoutes = router => {
  router.versioned.get({
    path: _constants2.OUTPUT_API_ROUTES.LIST_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ, _api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ]
        }]
      }
    },
    summary: 'Get outputs',
    options: {
      tags: ['oas-tag:Fleet outputs']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOutputsRequestSchema,
      response: {
        200: {
          body: () => _types.GetOutputsResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getOutputsHandler);
  router.versioned.get({
    path: _constants2.OUTPUT_API_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ, _api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ]
        }]
      }
    },
    summary: 'Get output',
    description: 'Get output by ID.',
    options: {
      tags: ['oas-tag:Fleet outputs']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOneOutputRequestSchema,
      response: {
        200: {
          body: () => _types.OutputResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getOneOuputHandler);
  router.versioned.put({
    path: _constants2.OUTPUT_API_ROUTES.UPDATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL, _api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL]
        }]
      }
    },
    summary: 'Update output',
    description: 'Update output by ID.',
    options: {
      tags: ['oas-tag:Fleet outputs']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PutOutputRequestSchema,
      response: {
        200: {
          body: () => _types.OutputResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.putOutputHandler);
  router.versioned.post({
    path: _constants2.OUTPUT_API_ROUTES.CREATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: 'Create output',
    options: {
      tags: ['oas-tag:Fleet outputs']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostOutputRequestSchema,
      response: {
        200: {
          body: () => _types.OutputResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.postOutputHandler);
  router.versioned.delete({
    path: _constants2.OUTPUT_API_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: 'Delete output',
    description: 'Delete output by ID.',
    options: {
      tags: ['oas-tag:Fleet outputs']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeleteOutputRequestSchema,
      response: {
        200: {
          body: () => _types.DeleteOutputResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        },
        404: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.deleteOutputHandler);
  router.versioned.post({
    path: _constants2.OUTPUT_API_ROUTES.LOGSTASH_API_KEY_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: 'Generate a Logstash API key',
    options: {
      tags: ['oas-tag:Fleet outputs']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          body: () => _types.GenerateLogstashApiKeyResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.postLogstashApiKeyHandler);
  router.versioned.get({
    path: _constants2.OUTPUT_API_ROUTES.GET_OUTPUT_HEALTH_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ]
      }
    },
    summary: 'Get the latest output health',
    options: {
      tags: ['oas-tag:Fleet outputs']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetLatestOutputHealthRequestSchema,
      response: {
        200: {
          body: () => _types.GetLatestOutputHealthResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getLatestOutputHealth);
};
exports.registerRoutes = registerRoutes;