"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentUpgradeStatus = void 0;
exports.getDownloadEstimate = getDownloadEstimate;
exports.getUpgradeStartDelay = getUpgradeStartDelay;
var _react = _interopRequireWildcard(require("react"));
var _i18nReact = require("@kbn/i18n-react");
var _eui = require("@elastic/eui");
var _moment = _interopRequireDefault(require("moment"));
var _services = require("../../../../../../../common/services");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns a user-friendly string for the estimated remaining time until the upgrade is scheduled.
 */
function getUpgradeStartDelay(scheduledAt) {
  const timeDiffMillis = Date.parse(scheduledAt || '') - Date.now();
  if (timeDiffMillis < 0) {
    // The scheduled time should not be in the past, this would indicate an issue.
    // We choose not to provide a time estimation rather than negative duration.
    return '';
  }
  if (timeDiffMillis < 15 * 6e4) {
    return ' The upgrade will start in less than 15 minutes.';
  }
  if (timeDiffMillis < 30 * 6e4) {
    return ' The upgrade will start in less than 30 minutes.';
  }
  if (timeDiffMillis < 60 * 6e4) {
    return ' The upgrade will start in less than 1 hour.';
  }
  return ` The upgrade will start in less than ${Math.ceil(timeDiffMillis / 36e5)} hours.`;
}
function getDownloadEstimate(metadata) {
  if (!metadata || metadata.download_percent === undefined && metadata.download_rate === undefined) {
    return '';
  }
  let tooltip = '';
  if (metadata.download_percent !== undefined) {
    tooltip = `${metadata.download_percent}%`;
  }
  if (metadata.download_rate !== undefined) {
    tooltip += ` at ${formatRate(metadata.download_rate)}`;
  }
  return ` (${tooltip.trim()})`;
}
const formatRate = downloadRate => {
  let i = 0;
  const byteUnits = [' Bps', ' kBps', ' MBps', ' GBps'];
  for (; i < byteUnits.length - 1; i++) {
    if (downloadRate < 1024) break;
    downloadRate = downloadRate / 1024;
  }
  return downloadRate.toFixed(1) + byteUnits[i];
};
const formatRetryUntil = retryUntil => {
  if (!retryUntil) return '';
  const eta = new Date(retryUntil).toISOString();
  const remainingTime = Date.parse(retryUntil) - Date.now();
  const duration = _moment.default.utc(_moment.default.duration(remainingTime, 'milliseconds').asMilliseconds()).format('HH:mm');
  return remainingTime > 0 ? `Retrying until: ${eta} (${duration} remaining)` : '';
};
function getStatusComponents(agentUpgradeDetails) {
  var _agentUpgradeDetails$, _agentUpgradeDetails$2, _agentUpgradeDetails$5;
  switch (agentUpgradeDetails === null || agentUpgradeDetails === void 0 ? void 0 : agentUpgradeDetails.state) {
    case 'UPG_REQUESTED':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "accent",
          iconType: "calendar"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeRequested",
          defaultMessage: "Upgrade requested"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeRequested",
          defaultMessage: "The agent has requested an upgrade."
        })
      };
    case 'UPG_SCHEDULED':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "accent",
          iconType: "clock"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeScheduled",
          defaultMessage: "Upgrade scheduled"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeScheduled",
          defaultMessage: "The agent has been instructed to upgrade.{upgradeStartDelay}",
          values: {
            upgradeStartDelay: getUpgradeStartDelay((_agentUpgradeDetails$ = agentUpgradeDetails.metadata) === null || _agentUpgradeDetails$ === void 0 ? void 0 : _agentUpgradeDetails$.scheduled_at)
          }
        })
      };
    case 'UPG_DOWNLOADING':
      if (agentUpgradeDetails !== null && agentUpgradeDetails !== void 0 && (_agentUpgradeDetails$2 = agentUpgradeDetails.metadata) !== null && _agentUpgradeDetails$2 !== void 0 && _agentUpgradeDetails$2.retry_error_msg) {
        var _agentUpgradeDetails$3, _agentUpgradeDetails$4;
        return {
          Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
            color: "accent",
            iconType: "download"
          }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
            id: "xpack.fleet.agentUpgradeStatusBadge.upgradeDownloading",
            defaultMessage: "Upgrade downloading"
          })),
          WarningTooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
            id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeDownloadingFailed",
            defaultMessage: "Upgrade failing: {retryMsg}. {retryUntil}",
            values: {
              retryMsg: agentUpgradeDetails === null || agentUpgradeDetails === void 0 ? void 0 : (_agentUpgradeDetails$3 = agentUpgradeDetails.metadata) === null || _agentUpgradeDetails$3 === void 0 ? void 0 : _agentUpgradeDetails$3.retry_error_msg,
              retryUntil: formatRetryUntil(agentUpgradeDetails === null || agentUpgradeDetails === void 0 ? void 0 : (_agentUpgradeDetails$4 = agentUpgradeDetails.metadata) === null || _agentUpgradeDetails$4 === void 0 ? void 0 : _agentUpgradeDetails$4.retry_until)
            }
          })
        };
      }
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "accent",
          iconType: "download"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeDownloading",
          defaultMessage: "Upgrade downloading"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeDownloading",
          defaultMessage: "Downloading the new agent artifact version{downloadEstimate}.",
          values: {
            downloadEstimate: getDownloadEstimate(agentUpgradeDetails === null || agentUpgradeDetails === void 0 ? void 0 : agentUpgradeDetails.metadata)
          }
        })
      };
    case 'UPG_EXTRACTING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "accent",
          iconType: "package"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeExtracting",
          defaultMessage: "Upgrade extracting"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeExtracting",
          defaultMessage: "The new agent artifact is extracting."
        })
      };
    case 'UPG_REPLACING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "warning",
          iconType: "copy"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeReplacing",
          defaultMessage: "Upgrade replacing"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeReplacing",
          defaultMessage: "Replacing the agent artifact version."
        })
      };
    case 'UPG_RESTARTING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "warning",
          iconType: "refresh"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeRestarting",
          defaultMessage: "Upgrade restarting"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeRestarting",
          defaultMessage: "The agent is restarting to apply the update."
        })
      };
    case 'UPG_WATCHING':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "warning",
          iconType: "inspect"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeMonitoring",
          defaultMessage: "Upgrade monitoring"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeMonitoring",
          defaultMessage: "Monitoring the new agent version for errors."
        })
      };
    case 'UPG_ROLLBACK':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "danger",
          iconType: "returnKey"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeRolledBack",
          defaultMessage: "Upgrade rolled back"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeRolledBack",
          defaultMessage: "Upgrade unsuccessful. Rolling back to previous version."
        })
      };
    case 'UPG_FAILED':
      return {
        Badge: /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
          color: "danger",
          iconType: "error"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusBadge.upgradeFailed",
          defaultMessage: "Upgrade failed"
        })),
        TooltipText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeFailed",
          defaultMessage: "Upgrade failed: {errorMsg}.",
          values: {
            errorMsg: agentUpgradeDetails === null || agentUpgradeDetails === void 0 ? void 0 : (_agentUpgradeDetails$5 = agentUpgradeDetails.metadata) === null || _agentUpgradeDetails$5 === void 0 ? void 0 : _agentUpgradeDetails$5.error_msg
          }
        })
      };
    default:
      return null;
  }
}
const AgentUpgradeStatus = ({
  isAgentUpgradable,
  agent,
  latestAgentVersion
}) => {
  const isAgentUpgrading = (0, _react.useMemo)(() => agent.upgrade_started_at && !agent.upgraded_at, [agent.upgrade_started_at, agent.upgraded_at]);
  const status = (0, _react.useMemo)(() => getStatusComponents(agent.upgrade_details), [agent.upgrade_details]);
  const minVersion = '8.12';
  const notUpgradeableMessage = (0, _services.getNotUpgradeableMessage)(agent, latestAgentVersion);
  if (isAgentUpgradable && (0, _services.isAgentUpgradeAvailable)(agent, latestAgentVersion)) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
      color: "hollow",
      iconType: "sortUp"
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.agentUpgradeStatusBadge.upgradeAvailable",
      defaultMessage: "Upgrade available"
    }));
  }
  if (agent.upgrade_details && status) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
      gutterSize: "s",
      alignItems: "center",
      responsive: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, status.Badge), status.TooltipText && /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
      type: "iInCircle",
      content: status.TooltipText,
      color: "subdued"
    })), status.WarningTooltipText && /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
      type: "warning",
      content: status.WarningTooltipText,
      color: "warning"
    })));
  }
  if (isAgentUpgrading) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
      type: "iInCircle",
      content: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.fleet.agentUpgradeStatusTooltip.upgradeDetailsNotAvailable",
        defaultMessage: "Detailed upgrade status is available for Elastic Agents on version {minVersion} and higher.",
        values: {
          minVersion
        }
      }),
      color: "subdued"
    });
  }
  if (!isAgentUpgradable && notUpgradeableMessage) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
      type: "iInCircle",
      content: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.fleet.agentUpgradeStatusBadge.notUpgradeable",
        defaultMessage: "Agent not upgradeable: {reason}",
        values: {
          reason: notUpgradeableMessage
        }
      }),
      color: "subdued"
    });
  }
  return null;
};
exports.AgentUpgradeStatus = AgentUpgradeStatus;