"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectVariables = collectVariables;
exports.excludeVariablesFromCurrentCommand = excludeVariablesFromCurrentCommand;
var _visitor = require("@kbn/esql-ast/src/visitor");
var _constants = require("./constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function addToVariableOccurrences(variables, instance) {
  if (!variables.has(instance.name)) {
    variables.set(instance.name, []);
  }
  const variablesOccurrencies = variables.get(instance.name);
  variablesOccurrencies.push(instance);
}
function addToVariables(oldArg, newArg, fields, variables) {
  if ((0, _helpers.isColumnItem)(oldArg) && (0, _helpers.isColumnItem)(newArg)) {
    const newVariable = {
      name: newArg.parts.join('.'),
      type: 'double' /* fallback to number */,
      location: newArg.location
    };
    // Now workout the exact type
    // it can be a rename of another variable as well
    const oldRef = fields.get(oldArg.parts.join('.')) || variables.get(oldArg.parts.join('.'));
    if (oldRef) {
      addToVariableOccurrences(variables, newVariable);
      newVariable.type = Array.isArray(oldRef) ? oldRef[0].type : oldRef.type;
    }
  }
}
function excludeVariablesFromCurrentCommand(commands, currentCommand, fieldsMap, queryString) {
  const anyVariables = collectVariables(commands, fieldsMap, queryString);
  const currentCommandVariables = collectVariables([currentCommand], fieldsMap, queryString);
  const resultVariables = new Map();
  anyVariables.forEach((value, key) => {
    if (!currentCommandVariables.has(key)) {
      resultVariables.set(key, value);
    }
  });
  return resultVariables;
}
function addVariableFromAssignment(assignOperation, variables, fields) {
  if ((0, _helpers.isColumnItem)(assignOperation.args[0])) {
    const rightHandSideArgType = (0, _helpers.getExpressionType)(assignOperation.args[1], fields, variables);
    addToVariableOccurrences(variables, {
      name: assignOperation.args[0].parts.join('.'),
      type: rightHandSideArgType /* fallback to number */,
      location: assignOperation.args[0].location
    });
  }
}
function addVariableFromExpression(expressionOperation, queryString, variables, fields) {
  if (!expressionOperation.text.includes(_constants.EDITOR_MARKER)) {
    const expressionText = queryString.substring(expressionOperation.location.min, expressionOperation.location.max + 1);
    const expressionType = (0, _helpers.getExpressionType)(expressionOperation, fields, variables);
    addToVariableOccurrences(variables, {
      name: expressionText,
      type: expressionType,
      location: expressionOperation.location
    });
  }
}
function collectVariables(ast, fields, queryString) {
  const variables = new Map();
  const visitor = new _visitor.Visitor().on('visitLiteralExpression', ctx => {
    // TODO - add these as variables
  }).on('visitExpression', _ctx => {}) // required for the types :shrug:
  .on('visitRenameExpression', ctx => {
    const [oldArg, newArg] = ctx.node.args;
    addToVariables(oldArg, newArg, fields, variables);
  }).on('visitFunctionCallExpression', ctx => {
    const node = ctx.node;
    if (node.subtype === 'binary-expression' && node.name === 'where') {
      ctx.visitArgument(0, undefined);
      return;
    }
    if (node.name === '=') {
      addVariableFromAssignment(node, variables, fields);
    } else {
      addVariableFromExpression(node, queryString, variables, fields);
    }
  }).on('visitCommandOption', ctx => {
    if (ctx.node.name === 'by') {
      return [...ctx.visitArguments()];
    } else if (ctx.node.name === 'with') {
      for (const assignFn of ctx.node.args) {
        if ((0, _helpers.isFunctionItem)(assignFn)) {
          const [newArg, oldArg] = (assignFn === null || assignFn === void 0 ? void 0 : assignFn.args) || [];
          // TODO why is oldArg an array?
          if (Array.isArray(oldArg)) {
            addToVariables(oldArg[0], newArg, fields, variables);
          }
        }
      }
    }
  }).on('visitCommand', ctx => {
    const ret = [];
    if (['row', 'eval', 'stats', 'inlinestats', 'metrics', 'rename'].includes(ctx.node.name)) {
      ret.push(...ctx.visitArgs());
    }
    if (['stats', 'inlinestats', 'enrich'].includes(ctx.node.name)) {
      // BY and WITH can contain variables
      ret.push(...ctx.visitOptions());
    }
    return ret;
  }).on('visitQuery', ctx => [...ctx.visitCommands()]);
  visitor.visitQuery(ast);
  return variables;
}