"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readTypeDefinitionById = readTypeDefinitionById;
exports.readTypeDefinitions = readTypeDefinitions;
exports.storeTypeDefinition = storeTypeDefinition;
var _constants = require("../constants");
var _run_esql_query = require("../run_esql_query");
var _entity_definition_conflict = require("../errors/entity_definition_conflict");
var _unknown_entity_type = require("../errors/unknown_entity_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function storeTypeDefinition({
  type,
  clusterClient,
  logger,
  replace = false
}) {
  const esClient = clusterClient.asInternalUser;
  const types = await (0, _run_esql_query.runESQLQuery)('fetch type definition for conflict check', {
    esClient,
    query: `FROM ${_constants.DEFINITIONS_ALIAS} METADATA _id | WHERE definition_type == "type" AND _id == "${type.id}" | KEEP _id | LIMIT 1000`,
    logger
  });
  if (types.length !== 0 && replace === false) {
    logger.debug(`Entity type definition with ID ${type.id} already exists`);
    throw new _entity_definition_conflict.EntityDefinitionConflict('type', type.id);
  }
  const definition = {
    template_version: _constants.TEMPLATE_VERSION,
    definition_type: 'type',
    type
  };
  logger.debug(`Installing entity type definition ${type.id}`);
  await esClient.index({
    index: _constants.DEFINITIONS_ALIAS,
    id: `${definition.type.id}`,
    document: definition,
    refresh: 'wait_for'
  });
  return definition.type;
}
async function readTypeDefinitions(clusterClient, logger) {
  const esClient = clusterClient.asInternalUser;
  const types = await (0, _run_esql_query.runESQLQuery)('fetch all type definitions', {
    esClient,
    query: `FROM ${_constants.DEFINITIONS_ALIAS} METADATA _source | WHERE definition_type == "type" | KEEP _source | LIMIT 1000`,
    logger
  });
  return types.map(storedTypeDefinition => storedTypeDefinition._source.type);
}
async function readTypeDefinitionById(id, clusterClient, logger) {
  const esClient = clusterClient.asInternalUser;
  const types = await (0, _run_esql_query.runESQLQuery)('fetch type definition by ID', {
    esClient,
    query: `FROM ${_constants.DEFINITIONS_ALIAS} METADATA _id,_source | WHERE definition_type == "type" AND _id == "${id}" | KEEP _source | LIMIT 1000`,
    logger
  });
  if (types.length === 0) {
    throw new _unknown_entity_type.UnknownEntityType(`Type with ID ${id} not found`);
  }
  return types[0]._source.type;
}