"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMlModelDeploymentStatus = void 0;
var _ml = require("../../../common/types/ml");
var _ml_model_deployment_common = require("./ml_model_deployment_common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getMlModelDeploymentStatus = async (modelName, trainedModelsProvider) => {
  var _modelStatsResponse$t;
  if (!trainedModelsProvider) {
    throw new Error('Machine Learning is not enabled');
  }
  const modelDetailsRequest = {
    include: 'definition_status',
    model_id: modelName
  };

  // get the model details to see if we're downloaded...
  try {
    const modelDetailsResponse = await trainedModelsProvider.getTrainedModels(modelDetailsRequest);
    if (!modelDetailsResponse || modelDetailsResponse.count === 0) {
      // no model? return no status
      return getDefaultStatusReturn(_ml.MlModelDeploymentState.NotDeployed, modelName);
    }
    const firstTrainedModelConfig = modelDetailsResponse.trained_model_configs ? modelDetailsResponse.trained_model_configs[0] : undefined;

    // are we downloaded?
    if (!firstTrainedModelConfig || !firstTrainedModelConfig.fully_defined) {
      // we're still downloading...
      return getDefaultStatusReturn(_ml.MlModelDeploymentState.Downloading, modelName);
    }
  } catch (error) {
    if (!(0, _ml_model_deployment_common.isNotFoundExceptionError)(error)) {
      throw error;
    }
    // not found? return a default
    return getDefaultStatusReturn(_ml.MlModelDeploymentState.NotDeployed, modelName);
  }
  const modelRequest = {
    model_id: modelName
  };
  const modelStatsResponse = await trainedModelsProvider.getTrainedModelsStats(modelRequest);
  if (!modelStatsResponse.trained_model_stats || modelStatsResponse.trained_model_stats.length < 1 || ((_modelStatsResponse$t = modelStatsResponse.trained_model_stats[0]) === null || _modelStatsResponse$t === void 0 ? void 0 : _modelStatsResponse$t.deployment_stats) === undefined) {
    // if we're here - we're downloaded, but not deployed if we can't find the stats
    return getDefaultStatusReturn(_ml.MlModelDeploymentState.Downloaded, modelName);
  }
  const modelDeployment = modelStatsResponse.trained_model_stats[0].deployment_stats;
  return {
    deploymentState: getMlModelDeploymentStateForStatus(modelDeployment === null || modelDeployment === void 0 ? void 0 : modelDeployment.allocation_status.state),
    modelId: modelName,
    nodeAllocationCount: (modelDeployment === null || modelDeployment === void 0 ? void 0 : modelDeployment.allocation_status.allocation_count) || 0,
    startTime: (modelDeployment === null || modelDeployment === void 0 ? void 0 : modelDeployment.start_time) || 0,
    targetAllocationCount: (modelDeployment === null || modelDeployment === void 0 ? void 0 : modelDeployment.allocation_status.target_allocation_count) || 0,
    threadsPerAllocation: (modelDeployment === null || modelDeployment === void 0 ? void 0 : modelDeployment.threads_per_allocation) || 0
  };
};
exports.getMlModelDeploymentStatus = getMlModelDeploymentStatus;
function getDefaultStatusReturn(status, modelName) {
  return {
    deploymentState: status,
    modelId: modelName,
    nodeAllocationCount: 0,
    startTime: 0,
    targetAllocationCount: 0,
    threadsPerAllocation: 0
  };
}
function getMlModelDeploymentStateForStatus(state) {
  if (!state) {
    return _ml.MlModelDeploymentState.NotDeployed;
  }
  switch (state) {
    case 'starting':
      return _ml.MlModelDeploymentState.Starting;
    case 'started':
      return _ml.MlModelDeploymentState.Started;
    case 'fully_allocated':
      return _ml.MlModelDeploymentState.FullyAllocated;
  }

  // unknown state? return default
  return _ml.MlModelDeploymentState.NotDeployed;
}