"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createApiLogic = void 0;
var _kea = require("kea");
var _api = require("../../../../common/types/api");
var _flash_messages = require("../flash_messages");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_CREATE_API_OPTIONS = {
  clearFlashMessagesOnMakeRequest: true,
  showErrorFlash: true
};
const createApiLogic = (path, apiFunction, incomingOptions = {}) => {
  const options = {
    ...DEFAULT_CREATE_API_OPTIONS,
    ...incomingOptions
  };
  return (0, _kea.kea)({
    actions: {
      apiError: error => error,
      apiReset: true,
      apiSuccess: result => result,
      makeRequest: args => args
    },
    listeners: ({
      actions
    }) => ({
      apiError: error => {
        if (options.showErrorFlash) {
          (0, _flash_messages.flashAPIErrors)(error);
        }
      },
      apiSuccess: result => {
        if (options.showSuccessFlashFn) {
          (0, _flash_messages.flashSuccessToast)(options.showSuccessFlashFn(result), {
            'aria-live': 'assertive',
            role: 'alert'
          });
        }
      },
      makeRequest: async (args, breakpoint) => {
        if (options.clearFlashMessagesOnMakeRequest) {
          (0, _flash_messages.clearFlashMessages)();
        }
        if (options.requestBreakpointMS) {
          await breakpoint(options.requestBreakpointMS);
        }
        try {
          const result = await apiFunction(args);
          actions.apiSuccess(result);
        } catch (e) {
          actions.apiError(e);
        }
      }
    }),
    path: ['enterprise_search', 'api', ...path],
    // @ts-expect-error upgrade typescript v5.1.6
    reducers: () => ({
      apiStatus: [{
        status: _api.Status.IDLE
      }, {
        // @ts-expect-error upgrade typescript v5.1.6
        apiError: (_, error) => ({
          error,
          status: _api.Status.ERROR
        }),
        apiReset: () => ({
          status: _api.Status.IDLE
        }),
        // @ts-expect-error upgrade typescript v5.1.6
        apiSuccess: (_, data) => ({
          data,
          status: _api.Status.SUCCESS
        }),
        // @ts-expect-error upgrade typescript v5.1.6
        makeRequest: ({
          data
        }) => {
          return {
            data,
            status: _api.Status.LOADING
          };
        }
      }]
    }),
    selectors: ({
      selectors
    }) => ({
      data: [() => [selectors.apiStatus], apiStatus => apiStatus.data],
      error: [() => [selectors.apiStatus], apiStatus => apiStatus.error],
      status: [() => [selectors.apiStatus], apiStatus => apiStatus.status]
    })
  });
};
exports.createApiLogic = createApiLogic;