"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformBulkError = exports.isOpenSourceModel = exports.getLlmType = exports.getLlmClass = exports.createBulkErrorObject = exports.convertToSnakeCase = exports.buildRouteValidation = exports.buildResponse = exports.CustomResponseFactory = void 0;
var _fp = require("lodash/fp");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _server = require("@kbn/langchain/server");
var _constants = require("@kbn/stack-connectors-plugin/common/openai/constants");
var _custom_http_request_error = require("./custom_http_request_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createBulkErrorObject = ({
  id,
  statusCode,
  message
}) => {
  if (id != null) {
    return {
      id,
      error: {
        status_code: statusCode,
        message
      }
    };
  } else if (id != null) {
    return {
      id,
      error: {
        status_code: statusCode,
        message
      }
    };
  } else {
    return {
      id: '(unknown id)',
      error: {
        status_code: statusCode,
        message
      }
    };
  }
};
exports.createBulkErrorObject = createBulkErrorObject;
const transformBulkError = (id, err) => {
  if (err instanceof _custom_http_request_error.CustomHttpRequestError) {
    var _err$statusCode;
    return createBulkErrorObject({
      id,
      statusCode: (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 400,
      message: err.message
    });
  } else if (err instanceof _securitysolutionEsUtils.BadRequestError) {
    return createBulkErrorObject({
      id,
      statusCode: 400,
      message: err.message
    });
  } else {
    var _err$statusCode2;
    return createBulkErrorObject({
      id,
      statusCode: (_err$statusCode2 = err.statusCode) !== null && _err$statusCode2 !== void 0 ? _err$statusCode2 : 500,
      message: err.message
    });
  }
};
exports.transformBulkError = transformBulkError;
const buildRouteValidation = schema => (payload, {
  ok,
  badRequest
}) => {
  const {
    value,
    error
  } = schema.validate(payload);
  if (error) {
    return badRequest(error.message);
  }
  return ok(value);
};
exports.buildRouteValidation = buildRouteValidation;
const statusToErrorMessage = statusCode => {
  switch (statusCode) {
    case 400:
      return 'Bad Request';
    case 401:
      return 'Unauthorized';
    case 403:
      return 'Forbidden';
    case 404:
      return 'Not Found';
    case 409:
      return 'Conflict';
    case 500:
      return 'Internal Error';
    default:
      return '(unknown error)';
  }
};
class CustomResponseFactory {
  constructor(response) {
    this.response = response;
  }
  error({
    statusCode,
    body,
    headers
  }) {
    const contentType = {
      'content-type': 'application/json'
    };
    const defaultedHeaders = {
      ...contentType,
      ...(headers !== null && headers !== void 0 ? headers : {})
    };
    return this.response.custom({
      headers: defaultedHeaders,
      statusCode,
      body: Buffer.from(JSON.stringify({
        message: body !== null && body !== void 0 ? body : statusToErrorMessage(statusCode),
        status_code: statusCode
      }))
    });
  }
}
exports.CustomResponseFactory = CustomResponseFactory;
const buildResponse = response => new CustomResponseFactory(response);
exports.buildResponse = buildResponse;
const convertToSnakeCase = obj => {
  if (!obj) {
    return null;
  }
  return Object.keys(obj).reduce((acc, item) => {
    const newKey = (0, _fp.snakeCase)(item);
    return {
      ...acc,
      [newKey]: obj[item]
    };
  }, {});
};

/**
 * Returns the LangChain `llmType` for the given actionTypeId
 */
exports.convertToSnakeCase = convertToSnakeCase;
const getLlmType = actionTypeId => {
  const llmTypeDictionary = {
    [`.gen-ai`]: `openai`,
    [`.bedrock`]: `bedrock`,
    [`.gemini`]: `gemini`,
    [`.inference`]: `inference`
  };
  return llmTypeDictionary[actionTypeId];
};
exports.getLlmType = getLlmType;
const getLlmClass = llmType => {
  switch (llmType) {
    case 'bedrock':
      return _server.ActionsClientChatBedrockConverse;
    case 'gemini':
      return _server.ActionsClientChatVertexAI;
    case 'openai':
    default:
      return _server.ActionsClientChatOpenAI;
  }
};
exports.getLlmClass = getLlmClass;
const isOpenSourceModel = connector => {
  var _connector$config, _connector$config2, _connector$config3;
  if (connector == null) {
    return false;
  }
  const llmType = getLlmType(connector.actionTypeId);
  const isOpenAiType = llmType === 'openai';
  if (!isOpenAiType) {
    return false;
  }
  const connectorApiProvider = (_connector$config = connector.config) !== null && _connector$config !== void 0 && _connector$config.apiProvider ? (_connector$config2 = connector.config) === null || _connector$config2 === void 0 ? void 0 : _connector$config2.apiProvider : undefined;
  if (connectorApiProvider === _constants.OpenAiProviderType.Other) {
    return true;
  }
  const connectorApiUrl = (_connector$config3 = connector.config) !== null && _connector$config3 !== void 0 && _connector$config3.apiUrl ? connector.config.apiUrl : undefined;
  return !!connectorApiUrl && connectorApiUrl !== _constants.OPENAI_CHAT_URL && connectorApiProvider !== _constants.OpenAiProviderType.AzureAi;
};
exports.isOpenSourceModel = isOpenSourceModel;