"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.invokeAttackDiscoveryGraph = void 0;
var _server = require("@kbn/langchain/server");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _default_attack_discovery_graph = require("../../../../../lib/attack_discovery/graphs/default_attack_discovery_graph");
var _constants = require("../../../../../lib/attack_discovery/graphs/default_attack_discovery_graph/constants");
var _throw_if_error_counts_exceeded = require("../throw_if_error_counts_exceeded");
var _throw_if_invalid_anonymization = require("../throw_if_invalid_anonymization");
var _utils = require("../../../../utils");
var _prompts = require("../../../../../lib/attack_discovery/graphs/default_attack_discovery_graph/nodes/helpers/prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const invokeAttackDiscoveryGraph = async ({
  actionsClient,
  alertsIndexPattern,
  anonymizationFields,
  apiConfig,
  connectorTimeout,
  end,
  esClient,
  filter,
  langSmithProject,
  langSmithApiKey,
  latestReplacements,
  logger,
  onNewReplacements,
  savedObjectsClient,
  size,
  start
}) => {
  var _traceOptions$tracers;
  (0, _throw_if_invalid_anonymization.throwIfInvalidAnonymization)(anonymizationFields);
  const llmType = (0, _utils.getLlmType)(apiConfig.actionTypeId);
  const model = apiConfig.model;
  const tags = [_constants.ATTACK_DISCOVERY_TAG, llmType, model].flatMap(tag => tag !== null && tag !== void 0 ? tag : []);
  const traceOptions = {
    projectName: langSmithProject,
    tracers: [...(0, _langsmith.getLangSmithTracer)({
      apiKey: langSmithApiKey,
      projectName: langSmithProject,
      logger
    })]
  };
  const llm = new _server.ActionsClientLlm({
    actionsClient,
    connectorId: apiConfig.connectorId,
    llmType,
    logger,
    model,
    temperature: 0,
    // zero temperature for attack discovery, because we want structured JSON output
    timeout: connectorTimeout,
    traceOptions,
    telemetryMetadata: {
      pluginId: 'security_attack_discovery'
    }
  });
  if (llm == null) {
    throw new Error('LLM is required for attack discoveries');
  }
  const attackDiscoveryPrompts = await (0, _prompts.getAttackDiscoveryPrompts)({
    actionsClient,
    connectorId: apiConfig.connectorId,
    // if in future oss has different prompt, add it as model here
    model,
    provider: llmType,
    savedObjectsClient
  });
  const graph = (0, _default_attack_discovery_graph.getDefaultAttackDiscoveryGraph)({
    alertsIndexPattern,
    anonymizationFields,
    end,
    esClient,
    filter,
    llm,
    logger,
    onNewReplacements,
    prompts: attackDiscoveryPrompts,
    replacements: latestReplacements,
    size,
    start
  });
  logger === null || logger === void 0 ? void 0 : logger.debug(() => 'invokeAttackDiscoveryGraph: invoking the Attack discovery graph');
  const result = await graph.invoke({}, {
    callbacks: [...((_traceOptions$tracers = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.tracers) !== null && _traceOptions$tracers !== void 0 ? _traceOptions$tracers : [])],
    runName: _constants.ATTACK_DISCOVERY_GRAPH_RUN_NAME,
    tags
  });
  const {
    attackDiscoveries,
    anonymizedAlerts,
    errors,
    generationAttempts,
    hallucinationFailures,
    maxGenerationAttempts,
    maxHallucinationFailures
  } = result;
  (0, _throw_if_error_counts_exceeded.throwIfErrorCountsExceeded)({
    errors,
    generationAttempts,
    hallucinationFailures,
    logger,
    maxGenerationAttempts,
    maxHallucinationFailures
  });
  return {
    anonymizedAlerts,
    attackDiscoveries
  };
};
exports.invokeAttackDiscoveryGraph = invokeAttackDiscoveryGraph;