"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToUpdateScheme = exports.appendConversationMessages = void 0;
var _get_conversation = require("./get_conversation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const appendConversationMessages = async ({
  esClient,
  logger,
  conversationIndex,
  existingConversation,
  messages
}) => {
  var _existingConversation;
  const updatedAt = new Date().toISOString();
  const params = transformToUpdateScheme(updatedAt, [...((_existingConversation = existingConversation.messages) !== null && _existingConversation !== void 0 ? _existingConversation : []), ...messages]);
  try {
    var _existingConversation2;
    const response = await esClient.updateByQuery({
      conflicts: 'proceed',
      index: conversationIndex,
      query: {
        ids: {
          values: [(_existingConversation2 = existingConversation.id) !== null && _existingConversation2 !== void 0 ? _existingConversation2 : '']
        }
      },
      refresh: true,
      script: {
        lang: 'painless',
        params: {
          ...params
        },
        source: `
          if (params.assignEmpty == true || params.containsKey('messages')) {
            def messages = [];
            for (message in params.messages) {
              def newMessage = [:];
              newMessage['@timestamp'] = message['@timestamp'];
              newMessage.content = message.content;
              newMessage.is_error = message.is_error;
              newMessage.reader = message.reader;
              newMessage.role = message.role;
              if (message.trace_data != null) {
                newMessage.trace_data = message.trace_data;
              }
              if (message.metadata != null) {
                newMessage.metadata = message.metadata;
              }
              messages.add(newMessage);
            }
            ctx._source.messages = messages;
          }
          ctx._source.updated_at = params.updated_at;
        `
      }
    });
    if (response.failures && response.failures.length > 0) {
      logger.error(`Error appending conversation messages: ${response.failures.map(f => f.id)} for conversation by ID: ${existingConversation.id}`);
      return null;
    }
    const updatedConversation = await (0, _get_conversation.getConversation)({
      esClient,
      conversationIndex,
      id: existingConversation.id,
      logger
    });
    return updatedConversation;
  } catch (err) {
    logger.error(`Error appending conversation messages: ${err} for conversation by ID: ${existingConversation.id}`);
    throw err;
  }
};
exports.appendConversationMessages = appendConversationMessages;
const transformToUpdateScheme = (updatedAt, messages) => {
  return {
    updated_at: updatedAt,
    messages: messages === null || messages === void 0 ? void 0 : messages.map(message => ({
      '@timestamp': message.timestamp,
      content: message.content,
      is_error: message.isError,
      reader: message.reader,
      role: message.role,
      ...(message.metadata ? {
        metadata: {
          ...(message.metadata.contentReferences ? {
            content_references: message.metadata.contentReferences
          } : {})
        }
      } : {}),
      ...(message.traceData ? {
        trace_data: {
          trace_id: message.traceData.traceId,
          transaction_id: message.traceData.transactionId
        }
      } : {})
    }))
  };
};
exports.transformToUpdateScheme = transformToUpdateScheme;