"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.copySavedSearch = copySavedSearch;
exports.getDefaultAppState = getDefaultAppState;
exports.getSavedSearchContainer = getSavedSearchContainer;
exports.isEqualSavedSearch = isEqualSavedSearch;
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _public = require("@kbn/unified-histogram-plugin/public");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../common/constants");
var _restore_from_saved_search = require("../../../services/saved_searches/restore_from_saved_search");
var _update_saved_search = require("./utils/update_saved_search");
var _add_log = require("../../../utils/add_log");
var _state_helpers = require("../../../utils/state_helpers");
var _discover_app_state_container = require("./discover_app_state_container");
var _get_state_defaults = require("./utils/get_state_defaults");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const FILTERS_COMPARE_OPTIONS = {
  ..._esQuery.COMPARE_ALL_OPTIONS,
  state: false // We don't compare filter types (global vs appState).
};

/**
 * Container for the saved search state, allowing to load, update and persist the saved search
 * Can also be used to track changes to the saved search
 * It centralizes functionality that was spread across the Discover main codebase
 * There are 2 hooks to access the state of the saved search in React components:
 * - useSavedSearch for the current state, that's updated on every relevant state change
 * - useSavedSearchInitial for the persisted or initial state, just updated when the saved search is peristed or loaded
 */

function getSavedSearchContainer({
  services,
  globalStateContainer,
  internalStateContainer
}) {
  const initialSavedSearch = services.savedSearch.getNew();
  const savedSearchInitial$ = new _rxjs.BehaviorSubject(initialSavedSearch);
  const savedSearchCurrent$ = new _rxjs.BehaviorSubject(copySavedSearch(initialSavedSearch));
  const hasChanged$ = new _rxjs.BehaviorSubject(false);
  const set = savedSearch => {
    (0, _add_log.addLog)('[savedSearch] set', savedSearch);
    hasChanged$.next(false);
    savedSearchCurrent$.next(savedSearch);
    savedSearchInitial$.next(copySavedSearch(savedSearch));
    return savedSearch;
  };
  const getState = () => savedSearchCurrent$.getValue();
  const getInitial$ = () => savedSearchInitial$;
  const getCurrent$ = () => savedSearchCurrent$;
  const getHasChanged$ = () => hasChanged$;
  const getTitle = () => savedSearchCurrent$.getValue().title;
  const getId = () => savedSearchCurrent$.getValue().id;
  const newSavedSearch = async nextDataView => {
    (0, _add_log.addLog)('[savedSearch] new', {
      nextDataView
    });
    const dataView = nextDataView !== null && nextDataView !== void 0 ? nextDataView : getState().searchSource.getField('index');
    const nextSavedSearch = services.savedSearch.getNew();
    nextSavedSearch.searchSource.setField('index', dataView);
    const newAppState = getDefaultAppState(nextSavedSearch, services);
    const nextSavedSearchToSet = (0, _update_saved_search.updateSavedSearch)({
      savedSearch: {
        ...nextSavedSearch
      },
      dataView,
      state: newAppState,
      globalStateContainer,
      services
    });
    return set(nextSavedSearchToSet);
  };
  const persist = async (nextSavedSearch, saveOptions) => {
    var _replacementDataView;
    (0, _add_log.addLog)('[savedSearch] persist', {
      nextSavedSearch,
      saveOptions
    });
    const dataView = nextSavedSearch.searchSource.getField('index');
    const profileDataViewIds = internalStateContainer.getState().defaultProfileAdHocDataViewIds;
    let replacementDataView;

    // If the Discover session is using a default profile ad hoc data view,
    // we copy it with a new ID to avoid conflicts with the profile defaults
    if (dataView !== null && dataView !== void 0 && dataView.id && profileDataViewIds.includes(dataView.id)) {
      var _dataView$name;
      const replacementSpec = {
        ...dataView.toSpec(),
        id: (0, _uuid.v4)(),
        name: _i18n.i18n.translate('discover.savedSearch.defaultProfileDataViewCopyName', {
          defaultMessage: '{dataViewName} ({discoverSessionTitle})',
          values: {
            dataViewName: (_dataView$name = dataView.name) !== null && _dataView$name !== void 0 ? _dataView$name : dataView.getIndexPattern(),
            discoverSessionTitle: nextSavedSearch.title
          }
        })
      };

      // Skip field list fetching since the existing data view already has the fields
      replacementDataView = await services.dataViews.create(replacementSpec, true);
    }
    (0, _update_saved_search.updateSavedSearch)({
      savedSearch: nextSavedSearch,
      globalStateContainer,
      services,
      useFilterAndQueryServices: true,
      dataView: replacementDataView
    });
    const currentFilters = nextSavedSearch.searchSource.getField('filter');

    // If the data view was replaced, we need to update the filter references
    if (dataView !== null && dataView !== void 0 && dataView.id && (_replacementDataView = replacementDataView) !== null && _replacementDataView !== void 0 && _replacementDataView.id && Array.isArray(currentFilters)) {
      nextSavedSearch.searchSource.setField('filter', (0, _esQuery.updateFilterReferences)(currentFilters, dataView.id, replacementDataView.id));
    }
    const id = await services.savedSearch.save(nextSavedSearch, saveOptions || {});
    if (id) {
      set(nextSavedSearch);
    }
    return {
      id
    };
  };
  const assignNextSavedSearch = ({
    nextSavedSearch
  }) => {
    const hasChanged = !isEqualSavedSearch(savedSearchInitial$.getValue(), nextSavedSearch);
    hasChanged$.next(hasChanged);
    savedSearchCurrent$.next(nextSavedSearch);
  };
  const updateWithFilterManagerFilters = () => {
    const nextSavedSearch = {
      ...getState()
    };
    nextSavedSearch.searchSource.setField('filter', (0, _lodash.cloneDeep)(services.filterManager.getFilters()));
    assignNextSavedSearch({
      nextSavedSearch
    });
    (0, _add_log.addLog)('[savedSearch] updateWithFilterManagerFilters done', nextSavedSearch);
    return nextSavedSearch;
  };
  const update = ({
    nextDataView,
    nextState,
    useFilterAndQueryServices
  }) => {
    (0, _add_log.addLog)('[savedSearch] update', {
      nextDataView,
      nextState
    });
    const previousSavedSearch = getState();
    const dataView = nextDataView ? nextDataView : previousSavedSearch.searchSource.getField('index');
    const nextSavedSearch = (0, _update_saved_search.updateSavedSearch)({
      savedSearch: {
        ...previousSavedSearch
      },
      dataView,
      state: nextState || {},
      globalStateContainer,
      services,
      useFilterAndQueryServices
    });
    assignNextSavedSearch({
      nextSavedSearch
    });
    (0, _add_log.addLog)('[savedSearch] update done', nextSavedSearch);
    return nextSavedSearch;
  };
  const updateTimeRange = () => {
    const previousSavedSearch = getState();
    if (!previousSavedSearch.timeRestore) {
      return;
    }
    const refreshInterval = services.timefilter.getRefreshInterval();
    const nextSavedSearch = {
      ...previousSavedSearch,
      timeRange: services.timefilter.getTime(),
      refreshInterval: {
        value: refreshInterval.value,
        pause: refreshInterval.pause
      }
    };
    assignNextSavedSearch({
      nextSavedSearch
    });
    (0, _add_log.addLog)('[savedSearch] updateWithTimeRange done', nextSavedSearch);
  };
  const updateVisContext = ({
    nextVisContext
  }) => {
    const previousSavedSearch = getState();
    const nextSavedSearch = {
      ...previousSavedSearch,
      visContext: nextVisContext
    };
    assignNextSavedSearch({
      nextSavedSearch
    });
    (0, _add_log.addLog)('[savedSearch] updateVisContext done', nextSavedSearch);
  };
  const load = async id => {
    (0, _add_log.addLog)('[savedSearch] load', {
      id
    });
    const loadedSavedSearch = await services.savedSearch.get(id);
    (0, _restore_from_saved_search.restoreStateFromSavedSearch)({
      savedSearch: loadedSavedSearch,
      timefilter: services.timefilter
    });
    return set(loadedSavedSearch);
  };
  return {
    getCurrent$,
    getHasChanged$,
    getId,
    getInitial$,
    getState,
    getTitle,
    load,
    new: newSavedSearch,
    persist,
    set,
    update,
    updateTimeRange,
    updateWithFilterManagerFilters,
    updateVisContext
  };
}

/**
 * Copies a saved search object, due to the stateful nature of searchSource it has to be copied with a dedicated function
 * @param savedSearch
 */
function copySavedSearch(savedSearch) {
  return {
    ...savedSearch,
    ...{
      searchSource: savedSearch.searchSource.createCopy()
    }
  };
}
function getDefaultAppState(savedSearch, services) {
  return (0, _state_helpers.handleSourceColumnState)((0, _get_state_defaults.getStateDefaults)({
    savedSearch,
    services
  }), services.uiSettings);
}
function isEqualSavedSearch(savedSearchPrev, savedSearchNext) {
  const {
    searchSource: prevSearchSource,
    ...prevSavedSearch
  } = savedSearchPrev;
  const {
    searchSource: nextSearchSource,
    ...nextSavedSearchWithoutSearchSource
  } = savedSearchNext;
  const keys = new Set([...Object.keys(prevSavedSearch), ...Object.keys(nextSavedSearchWithoutSearchSource)]);

  // at least one change in saved search attributes
  const hasChangesInSavedSearch = [...keys].some(key => {
    if (['usesAdHocDataView', 'hideChart'].includes(key) && typeof prevSavedSearch[key] === 'undefined' && nextSavedSearchWithoutSearchSource[key] === false) {
      return false; // ignore when value was changed from `undefined` to `false` as it happens per app logic, not by a user action
    }
    const prevValue = getSavedSearchFieldForComparison(prevSavedSearch, key);
    const nextValue = getSavedSearchFieldForComparison(nextSavedSearchWithoutSearchSource, key);
    const isSame = (0, _lodash.isEqual)(prevValue, nextValue);
    if (!isSame) {
      (0, _add_log.addLog)('[savedSearch] difference between initial and changed version', {
        key,
        before: prevSavedSearch[key],
        after: nextSavedSearchWithoutSearchSource[key]
      });
    }
    return !isSame;
  });
  if (hasChangesInSavedSearch) {
    return false;
  }

  // at least one change in search source fields
  const hasChangesInSearchSource = ['filter', 'query', 'index'].some(key => {
    const prevValue = getSearchSourceFieldValueForComparison(prevSearchSource, key);
    const nextValue = getSearchSourceFieldValueForComparison(nextSearchSource, key);
    const isSame = key === 'filter' ? (0, _discover_app_state_container.isEqualFilters)(prevValue, nextValue, FILTERS_COMPARE_OPTIONS) // if a filter gets pinned and the order of filters does not change, we don't show the unsaved changes badge
    : (0, _lodash.isEqual)(prevValue, nextValue);
    if (!isSame) {
      (0, _add_log.addLog)('[savedSearch] difference between initial and changed version', {
        key,
        before: prevValue,
        after: nextValue
      });
    }
    return !isSame;
  });
  if (hasChangesInSearchSource) {
    return false;
  }
  (0, _add_log.addLog)('[savedSearch] no difference between initial and changed version');
  return true;
}
function getSavedSearchFieldForComparison(savedSearch, fieldName) {
  if (fieldName === 'visContext') {
    var _visContext$attribute;
    const visContext = (0, _lodash.cloneDeep)(savedSearch.visContext);
    if ((0, _public.canImportVisContext)(visContext) && visContext !== null && visContext !== void 0 && (_visContext$attribute = visContext.attributes) !== null && _visContext$attribute !== void 0 && _visContext$attribute.title) {
      // ignore differences in title as it sometimes does not match the actual vis type/shape
      visContext.attributes.title = 'same';
    }
    return visContext;
  }
  if (fieldName === 'breakdownField') {
    return savedSearch.breakdownField || ''; // ignore the difference between an empty string and undefined
  }
  if (fieldName === 'viewMode') {
    var _savedSearch$viewMode;
    // By default, viewMode: undefined is equivalent to documents view
    // So they should be treated as same
    return (_savedSearch$viewMode = savedSearch.viewMode) !== null && _savedSearch$viewMode !== void 0 ? _savedSearch$viewMode : _constants.VIEW_MODE.DOCUMENT_LEVEL;
  }
  return savedSearch[fieldName];
}
function getSearchSourceFieldValueForComparison(searchSource, searchSourceFieldName) {
  if (searchSourceFieldName === 'index') {
    var _searchSource$getFiel;
    const query = searchSource.getField('query');
    // ad-hoc data view id can change, so we rather compare the ES|QL query itself here
    return query && 'esql' in query ? query.esql : (_searchSource$getFiel = searchSource.getField('index')) === null || _searchSource$getFiel === void 0 ? void 0 : _searchSource$getFiel.id;
  }
  if (searchSourceFieldName === 'filter') {
    const filterField = searchSource.getField('filter');
    return (0, _lodash.isFunction)(filterField) ? filterField() : filterField;
  }
  return searchSource.getField(searchSourceFieldName);
}