"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPureDatasetQualityDetailsControllerStateMachine = exports.createDatasetQualityDetailsControllerStateMachine = void 0;
var _timerange = require("@kbn/timerange");
var _xstate = require("xstate");
var _utils = require("../../../common/utils");
var _notifications = require("../common/notifications");
var _notifications2 = require("./notifications");
var _quality_issues = require("../../utils/quality_issues");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPureDatasetQualityDetailsControllerStateMachine = initialContext => (0, _xstate.createMachine)({
  id: 'DatasetQualityDetailsController',
  context: initialContext,
  predictableActionArguments: true,
  initial: 'initializing',
  states: {
    initializing: {
      type: 'parallel',
      states: {
        nonAggregatableDataset: {
          initial: 'fetching',
          states: {
            fetching: {
              invoke: {
                src: 'checkDatasetIsAggregatable',
                onDone: {
                  target: 'done',
                  actions: ['storeDatasetAggregatableStatus']
                },
                onError: [{
                  target: '#DatasetQualityDetailsController.indexNotFound',
                  cond: 'isIndexNotFoundError'
                }, {
                  target: 'done',
                  actions: ['notifyFailedFetchForAggregatableDatasets']
                }]
              }
            },
            done: {
              on: {
                UPDATE_TIME_RANGE: {
                  target: 'fetching',
                  actions: ['storeTimeRange']
                }
              }
            }
          }
        },
        dataStreamDetails: {
          initial: 'fetching',
          states: {
            fetching: {
              invoke: {
                src: 'loadDataStreamDetails',
                onDone: {
                  target: 'done',
                  actions: ['storeDataStreamDetails']
                },
                onError: [{
                  target: '#DatasetQualityDetailsController.indexNotFound',
                  cond: 'isIndexNotFoundError'
                }, {
                  target: 'done',
                  actions: ['notifyFetchDataStreamDetailsFailed']
                }]
              }
            },
            done: {
              on: {
                UPDATE_TIME_RANGE: {
                  target: 'fetching',
                  actions: ['storeTimeRange']
                },
                BREAKDOWN_FIELD_CHANGE: {
                  target: '#DatasetQualityDetailsController.initializing.checkBreakdownFieldIsEcs.fetching',
                  actions: ['storeBreakDownField']
                },
                QUALITY_ISSUES_CHART_CHANGE: {
                  target: 'done',
                  actions: ['storeQualityIssuesChart']
                }
              }
            }
          }
        },
        checkBreakdownFieldIsEcs: {
          initial: 'fetching',
          states: {
            fetching: {
              invoke: {
                src: 'checkBreakdownFieldIsEcs',
                onDone: {
                  target: 'done',
                  actions: ['storeBreakdownFieldEcsStatus']
                },
                onError: {
                  target: 'done',
                  actions: ['notifyCheckBreakdownFieldIsEcsFailed']
                }
              }
            },
            done: {}
          }
        },
        dataStreamSettings: {
          initial: 'fetchingDataStreamSettings',
          states: {
            fetchingDataStreamSettings: {
              invoke: {
                src: 'loadDataStreamSettings',
                onDone: {
                  target: 'qualityIssues',
                  actions: ['storeDataStreamSettings']
                },
                onError: [{
                  target: '#DatasetQualityDetailsController.indexNotFound',
                  cond: 'isIndexNotFoundError'
                }, {
                  target: 'errorFetchingDataStreamSettings',
                  actions: ['notifyFetchDataStreamSettingsFailed']
                }]
              }
            },
            errorFetchingDataStreamSettings: {},
            qualityIssues: {
              type: 'parallel',
              states: {
                dataStreamDegradedFields: {
                  initial: 'fetchingDataStreamDegradedFields',
                  states: {
                    fetchingDataStreamDegradedFields: {
                      invoke: {
                        src: 'loadDegradedFields',
                        onDone: {
                          target: 'doneFetchingDegradedFields',
                          actions: ['storeDegradedFields']
                        },
                        onError: [{
                          target: '#DatasetQualityDetailsController.indexNotFound',
                          cond: 'isIndexNotFoundError'
                        }, {
                          target: 'errorFetchingDegradedFields'
                        }]
                      }
                    },
                    errorFetchingDegradedFields: {},
                    doneFetchingDegradedFields: {
                      type: 'final'
                    }
                  }
                },
                dataStreamFailedDocs: {
                  initial: 'fetchingFailedDocs',
                  states: {
                    fetchingFailedDocs: {
                      invoke: {
                        src: 'loadFailedDocsDetails',
                        onDone: {
                          target: 'doneFetchingFailedDocs',
                          actions: ['storeFailedDocsDetails']
                        },
                        onError: [{
                          target: 'notImplemented',
                          cond: 'checkIfNotImplemented'
                        }, {
                          target: '#DatasetQualityDetailsController.indexNotFound',
                          cond: 'isIndexNotFoundError'
                        }, {
                          target: 'errorFetchingFailedDocs'
                        }]
                      }
                    },
                    notImplemented: {
                      type: 'final'
                    },
                    errorFetchingFailedDocs: {},
                    doneFetchingFailedDocs: {
                      type: 'final'
                    }
                  }
                }
              },
              onDone: {
                target: '#DatasetQualityDetailsController.initializing.dataStreamSettings.doneFetchingQualityIssues'
              }
            },
            doneFetchingQualityIssues: {
              entry: ['raiseDegradedFieldsLoaded'],
              on: {
                UPDATE_QUALITY_ISSUES_TABLE_CRITERIA: {
                  target: 'doneFetchingQualityIssues',
                  actions: ['storeQualityIssuesTableOptions']
                },
                OPEN_QUALITY_ISSUE_FLYOUT: {
                  target: '#DatasetQualityDetailsController.initializing.qualityIssueFlyout.open',
                  actions: ['storeExpandedQualityIssue', 'resetFieldLimitServerResponse']
                },
                TOGGLE_CURRENT_QUALITY_ISSUES: {
                  target: '#DatasetQualityDetailsController.initializing.dataStreamSettings.qualityIssues.dataStreamDegradedFields.fetchingDataStreamDegradedFields',
                  actions: ['toggleCurrentQualityIssues']
                }
              }
            }
          },
          on: {
            UPDATE_TIME_RANGE: {
              target: '.fetchingDataStreamSettings'
            }
          }
        },
        checkAndLoadIntegrationAndDashboards: {
          initial: 'checkingAndLoadingIntegration',
          states: {
            checkingAndLoadingIntegration: {
              invoke: {
                src: 'checkAndLoadIntegration',
                onDone: [{
                  target: 'loadingIntegrationDashboards',
                  actions: 'storeDataStreamIntegration',
                  cond: 'isDataStreamIsPartOfIntegration'
                }, {
                  actions: 'storeDataStreamIntegration',
                  target: 'done'
                }],
                onError: {
                  target: 'done',
                  actions: ['notifyFetchDatasetIntegrationsFailed']
                }
              }
            },
            loadingIntegrationDashboards: {
              invoke: {
                src: 'loadIntegrationDashboards',
                onDone: {
                  target: 'done',
                  actions: ['storeIntegrationDashboards']
                },
                onError: [{
                  target: 'unauthorizedToLoadDashboards',
                  cond: 'checkIfActionForbidden'
                }, {
                  target: 'done',
                  actions: ['notifyFetchIntegrationDashboardsFailed']
                }]
              }
            },
            unauthorizedToLoadDashboards: {
              type: 'final'
            },
            done: {}
          }
        },
        qualityIssueFlyout: {
          initial: 'pending',
          states: {
            pending: {
              always: [{
                target: 'closed',
                cond: 'hasNoQualityIssueSelected'
              }]
            },
            open: {
              initial: 'initializing',
              states: {
                initializing: {
                  always: [{
                    target: 'degradedFieldFlyout',
                    cond: 'isDegradedFieldFlyout'
                  }, {
                    target: 'failedDocsFlyout'
                  }]
                },
                failedDocsFlyout: {
                  initial: 'fetching',
                  states: {
                    fetching: {
                      invoke: {
                        src: 'loadfailedDocsErrors',
                        onDone: {
                          target: 'done',
                          actions: ['storeFailedDocsErrors']
                        },
                        onError: [{
                          target: 'unsupported',
                          cond: 'checkIfNotImplemented'
                        }, {
                          target: '#DatasetQualityDetailsController.indexNotFound',
                          cond: 'isIndexNotFoundError'
                        }, {
                          target: 'done'
                        }]
                      }
                    },
                    done: {
                      on: {
                        UPDATE_FAILED_DOCS_ERRORS_TABLE_CRITERIA: {
                          target: 'done',
                          actions: ['storeFailedDocsErrorsTableOptions']
                        }
                      }
                    },
                    unsupported: {}
                  }
                },
                degradedFieldFlyout: {
                  type: 'parallel',
                  states: {
                    ignoredValues: {
                      initial: 'fetching',
                      states: {
                        fetching: {
                          invoke: {
                            src: 'loadDegradedFieldValues',
                            onDone: {
                              target: 'done',
                              actions: ['storeDegradedFieldValues']
                            },
                            onError: [{
                              target: '#DatasetQualityDetailsController.indexNotFound',
                              cond: 'isIndexNotFoundError'
                            }, {
                              target: 'done'
                            }]
                          }
                        },
                        done: {}
                      }
                    },
                    mitigation: {
                      initial: 'analyzing',
                      states: {
                        analyzing: {
                          invoke: {
                            src: 'analyzeDegradedField',
                            onDone: {
                              target: 'analyzed',
                              actions: ['storeDegradedFieldAnalysis']
                            },
                            onError: {
                              target: 'analyzed'
                            }
                          }
                        },
                        analyzed: {
                          on: {
                            SET_NEW_FIELD_LIMIT: {
                              target: 'mitigating',
                              actions: 'storeNewFieldLimit'
                            }
                          }
                        },
                        mitigating: {
                          invoke: {
                            src: 'saveNewFieldLimit',
                            onDone: [{
                              target: 'askingForRollover',
                              actions: 'storeNewFieldLimitResponse',
                              cond: 'hasFailedToUpdateLastBackingIndex'
                            }, {
                              target: 'success',
                              actions: 'storeNewFieldLimitResponse'
                            }],
                            onError: {
                              target: 'error',
                              actions: ['storeNewFieldLimitErrorResponse', 'notifySaveNewFieldLimitError']
                            }
                          }
                        },
                        askingForRollover: {
                          on: {
                            ROLLOVER_DATA_STREAM: {
                              target: 'rollingOver'
                            }
                          }
                        },
                        rollingOver: {
                          invoke: {
                            src: 'rolloverDataStream',
                            onDone: {
                              target: 'success',
                              actions: ['raiseForceTimeRangeRefresh']
                            },
                            onError: {
                              target: 'error',
                              actions: 'notifySaveNewFieldLimitError'
                            }
                          }
                        },
                        success: {},
                        error: {}
                      }
                    }
                  }
                }
              },
              on: {
                CLOSE_DEGRADED_FIELD_FLYOUT: {
                  target: 'closed',
                  actions: ['storeExpandedQualityIssue']
                },
                UPDATE_TIME_RANGE: {
                  target: '#DatasetQualityDetailsController.initializing.qualityIssueFlyout.open'
                }
              }
            },
            closed: {
              on: {
                OPEN_QUALITY_ISSUE_FLYOUT: {
                  target: '#DatasetQualityDetailsController.initializing.qualityIssueFlyout.open',
                  actions: ['storeExpandedQualityIssue']
                }
              }
            }
          },
          on: {
            DEGRADED_FIELDS_LOADED: [{
              target: '.open',
              cond: 'shouldOpenFlyout'
            }, {
              target: '.closed',
              actions: ['storeExpandedQualityIssue']
            }]
          }
        }
      }
    },
    indexNotFound: {
      entry: 'handleIndexNotFoundError'
    }
  }
}, {
  actions: {
    storeDatasetAggregatableStatus: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        isNonAggregatable: !event.data.aggregatable
      } : {};
    }),
    storeTimeRange: (0, _xstate.assign)((context, event) => {
      return {
        timeRange: 'timeRange' in event ? event.timeRange : context.timeRange
      };
    }),
    storeDataStreamDetails: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        dataStreamDetails: event.data
      } : {};
    }),
    storeQualityIssuesChart: (0, _xstate.assign)((_context, event) => {
      return 'qualityIssuesChart' in event ? {
        qualityIssuesChart: event.qualityIssuesChart
      } : {};
    }),
    storeBreakDownField: (0, _xstate.assign)((_context, event) => {
      return 'breakdownField' in event ? {
        breakdownField: event.breakdownField
      } : {};
    }),
    storeBreakdownFieldEcsStatus: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        isBreakdownFieldEcs: event.data
      } : {};
    }),
    storeFailedDocsDetails: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        qualityIssues: {
          ...context.qualityIssues,
          data: [...(0, _quality_issues.filterIssues)(context.qualityIssues.data, 'failed'), ...(0, _quality_issues.mapFailedDocsIssues)(event.data)]
        }
      } : {};
    }),
    storeFailedDocsErrors: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        failedDocsErrors: {
          ...context.failedDocsErrors,
          data: event.data.errors
        }
      } : {};
    }),
    storeDegradedFields: (0, _xstate.assign)((context, event) => {
      var _event$data;
      return 'data' in event ? {
        qualityIssues: {
          ...context.qualityIssues,
          data: [...(0, _quality_issues.filterIssues)(context.qualityIssues.data, 'degraded'), ...(0, _quality_issues.mapDegradedFieldsIssues)((_event$data = event.data) === null || _event$data === void 0 ? void 0 : _event$data.degradedFields)]
        }
      } : {};
    }),
    storeDegradedFieldValues: (0, _xstate.assign)((_, event) => {
      return 'data' in event ? {
        degradedFieldValues: event.data
      } : {};
    }),
    storeDegradedFieldAnalysis: (0, _xstate.assign)((_, event) => {
      return 'data' in event ? {
        degradedFieldAnalysis: event.data
      } : {};
    }),
    storeQualityIssuesTableOptions: (0, _xstate.assign)((context, event) => {
      return 'quality_issues_criteria' in event ? {
        qualityIssues: {
          ...context.qualityIssues,
          table: event.quality_issues_criteria
        }
      } : {};
    }),
    storeFailedDocsErrorsTableOptions: (0, _xstate.assign)((context, event) => {
      return 'failed_docs_errors_criteria' in event ? {
        failedDocsErrors: {
          ...context.failedDocsErrors,
          table: event.failed_docs_errors_criteria
        }
      } : {};
    }),
    storeExpandedQualityIssue: (0, _xstate.assign)((_, event) => {
      return {
        expandedQualityIssue: 'qualityIssue' in event ? {
          name: event.qualityIssue.name,
          type: event.qualityIssue.type
        } : undefined
      };
    }),
    toggleCurrentQualityIssues: (0, _xstate.assign)(context => {
      return {
        showCurrentQualityIssues: !context.showCurrentQualityIssues
      };
    }),
    raiseDegradedFieldsLoaded: (0, _xstate.raise)('DEGRADED_FIELDS_LOADED'),
    storeDataStreamSettings: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        dataStreamSettings: event.data
      } : {};
    }),
    storeDataStreamIntegration: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        integration: event.data
      } : {};
    }),
    storeIntegrationDashboards: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        integrationDashboards: event.data
      } : {};
    }),
    handleIndexNotFoundError: (0, _xstate.assign)(() => {
      return {
        isIndexNotFoundError: true
      };
    }),
    storeNewFieldLimit: (0, _xstate.assign)((_, event) => {
      return 'newFieldLimit' in event ? {
        fieldLimit: {
          newFieldLimit: event.newFieldLimit
        }
      } : {};
    }),
    storeNewFieldLimitResponse: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        fieldLimit: {
          ...context.fieldLimit,
          result: event.data,
          error: false
        }
      } : {};
    }),
    storeNewFieldLimitErrorResponse: (0, _xstate.assign)(context => {
      return {
        fieldLimit: {
          ...context.fieldLimit,
          error: true
        }
      };
    }),
    resetFieldLimitServerResponse: (0, _xstate.assign)(() => ({
      fieldLimit: undefined
    })),
    raiseForceTimeRangeRefresh: (0, _xstate.raise)('UPDATE_TIME_RANGE')
  },
  guards: {
    checkIfActionForbidden: (_, event) => {
      return 'data' in event && typeof event.data === 'object' && 'statusCode' in event.data && event.data.statusCode === 403;
    },
    checkIfNotImplemented: (_context, event) => {
      return 'data' in event && typeof event.data === 'object' && 'statusCode' in event.data && event.data.statusCode === 501;
    },
    isIndexNotFoundError: (_, event) => {
      var _ref, _event$data$originalM;
      return (_ref = 'data' in event && typeof event.data === 'object' && 'statusCode' in event.data && event.data.statusCode === 500 && 'originalMessage' in event.data && ((_event$data$originalM = event.data.originalMessage) === null || _event$data$originalM === void 0 ? void 0 : _event$data$originalM.includes('index_not_found_exception'))) !== null && _ref !== void 0 ? _ref : false;
    },
    shouldOpenFlyout: (context, _event, meta) => {
      var _context$qualityIssue;
      return Boolean(context.expandedQualityIssue) && Boolean((_context$qualityIssue = context.qualityIssues.data) === null || _context$qualityIssue === void 0 ? void 0 : _context$qualityIssue.some(field => {
        var _context$expandedQual;
        return field.name === ((_context$expandedQual = context.expandedQualityIssue) === null || _context$expandedQual === void 0 ? void 0 : _context$expandedQual.name);
      }));
    },
    isDegradedFieldFlyout: context => {
      return Boolean(context.expandedQualityIssue && context.expandedQualityIssue.type === 'degraded');
    },
    hasNoQualityIssueSelected: context => {
      return !Boolean(context.expandedQualityIssue);
    },
    hasFailedToUpdateLastBackingIndex: (_, event) => {
      return 'data' in event && typeof event.data === 'object' && 'isLatestBackingIndexUpdated' in event.data && !event.data.isLatestBackingIndexUpdated;
    },
    isDataStreamIsPartOfIntegration: (_, event) => {
      return 'data' in event && typeof event.data === 'object' && 'isIntegration' in event.data && event.data.isIntegration;
    }
  }
});
exports.createPureDatasetQualityDetailsControllerStateMachine = createPureDatasetQualityDetailsControllerStateMachine;
const createDatasetQualityDetailsControllerStateMachine = ({
  initialContext,
  plugins,
  toasts,
  dataStreamStatsClient,
  dataStreamDetailsClient,
  isFailureStoreEnabled
}) => createPureDatasetQualityDetailsControllerStateMachine(initialContext).withConfig({
  actions: {
    notifyFailedFetchForAggregatableDatasets: (_context, event) => (0, _notifications.fetchNonAggregatableDatasetsFailedNotifier)(toasts, event.data),
    notifyFetchDataStreamDetailsFailed: (_context, event) => (0, _notifications2.fetchDataStreamDetailsFailedNotifier)(toasts, event.data),
    notifyCheckBreakdownFieldIsEcsFailed: (_context, event) => (0, _notifications2.assertBreakdownFieldEcsFailedNotifier)(toasts, event.data),
    notifyFetchDataStreamSettingsFailed: (_context, event) => (0, _notifications2.fetchDataStreamSettingsFailedNotifier)(toasts, event.data),
    notifyFetchIntegrationDashboardsFailed: (_context, event) => (0, _notifications2.fetchIntegrationDashboardsFailedNotifier)(toasts, event.data),
    notifyFetchDatasetIntegrationsFailed: (context, event) => {
      return (0, _notifications2.fetchDataStreamIntegrationFailedNotifier)(toasts, event.data);
    },
    notifySaveNewFieldLimitError: (_context, event) => (0, _notifications2.updateFieldLimitFailedNotifier)(toasts, event.data),
    notifyRolloverDataStreamError: (context, event) => (0, _notifications2.rolloverDataStreamFailedNotifier)(toasts, event.data, context.dataStream)
  },
  services: {
    checkDatasetIsAggregatable: context => {
      const {
        type
      } = (0, _utils.indexNameToDataStreamParts)(context.dataStream);
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.timeRange);
      return dataStreamStatsClient.getNonAggregatableDatasets({
        types: [type],
        start,
        end,
        dataStream: context.dataStream
      });
    },
    loadDataStreamDetails: context => {
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.timeRange);
      return dataStreamDetailsClient.getDataStreamDetails({
        dataStream: context.dataStream,
        start,
        end
      });
    },
    checkBreakdownFieldIsEcs: async context => {
      if (context.breakdownField) {
        var _fields$context$break;
        const allowedFieldSources = ['ecs', 'metadata'];

        // This timeout is to avoid a runtime error that randomly happens on breakdown field change
        // TypeError: Cannot read properties of undefined (reading 'timeFieldName')
        await new Promise(res => setTimeout(res, 300));
        const client = await plugins.fieldsMetadata.getClient();
        const {
          fields
        } = await client.find({
          attributes: ['source'],
          fieldNames: [context.breakdownField]
        });
        const breakdownFieldSource = (_fields$context$break = fields[context.breakdownField]) === null || _fields$context$break === void 0 ? void 0 : _fields$context$break.source;
        return !!(breakdownFieldSource && allowedFieldSources.includes(breakdownFieldSource));
      }
      return false;
    },
    loadFailedDocsDetails: context => {
      if (!isFailureStoreEnabled) {
        const unsupportedError = {
          message: 'Failure store is disabled',
          statusCode: 501
        };
        return Promise.reject(unsupportedError);
      }
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.timeRange);
      return dataStreamDetailsClient.getFailedDocsDetails({
        dataStream: context.dataStream,
        start,
        end
      });
    },
    loadDegradedFields: context => {
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.timeRange);
      if (!(context !== null && context !== void 0 && context.isNonAggregatable)) {
        return dataStreamDetailsClient.getDataStreamDegradedFields({
          dataStream: context.showCurrentQualityIssues && 'dataStreamSettings' in context && context.dataStreamSettings && context.dataStreamSettings.lastBackingIndexName ? context.dataStreamSettings.lastBackingIndexName : context.dataStream,
          start,
          end
        });
      }
      return Promise.resolve();
    },
    loadDegradedFieldValues: context => {
      if ('expandedQualityIssue' in context && context.expandedQualityIssue) {
        return dataStreamDetailsClient.getDataStreamDegradedFieldValues({
          dataStream: context.dataStream,
          degradedField: context.expandedQualityIssue.name
        });
      }
      return Promise.resolve();
    },
    analyzeDegradedField: context => {
      var _context$qualityIssue2, _context$qualityIssue3;
      if (context !== null && context !== void 0 && (_context$qualityIssue2 = context.qualityIssues) !== null && _context$qualityIssue2 !== void 0 && (_context$qualityIssue3 = _context$qualityIssue2.data) !== null && _context$qualityIssue3 !== void 0 && _context$qualityIssue3.length) {
        const selectedDegradedField = context.qualityIssues.data.find(field => {
          var _context$expandedQual2;
          return field.name === ((_context$expandedQual2 = context.expandedQualityIssue) === null || _context$expandedQual2 === void 0 ? void 0 : _context$expandedQual2.name);
        });
        if (selectedDegradedField && selectedDegradedField.type === 'degraded') {
          var _context$expandedQual3;
          return dataStreamDetailsClient.analyzeDegradedField({
            dataStream: context.dataStream,
            degradedField: (_context$expandedQual3 = context.expandedQualityIssue) === null || _context$expandedQual3 === void 0 ? void 0 : _context$expandedQual3.name,
            lastBackingIndex: selectedDegradedField.indexFieldWasLastPresentIn
          });
        }
      }
      return Promise.resolve();
    },
    loadfailedDocsErrors: context => {
      if (!isFailureStoreEnabled) {
        const unsupportedError = {
          message: 'Failure store is disabled',
          statusCode: 501
        };
        return Promise.reject(unsupportedError);
      }
      if ('expandedQualityIssue' in context && context.expandedQualityIssue) {
        const {
          startDate: start,
          endDate: end
        } = (0, _timerange.getDateISORange)(context.timeRange);
        return dataStreamDetailsClient.getFailedDocsErrors({
          dataStream: context.dataStream,
          start,
          end
        });
      }
      return Promise.resolve();
    },
    loadDataStreamSettings: context => {
      return dataStreamDetailsClient.getDataStreamSettings({
        dataStream: context.dataStream
      });
    },
    checkAndLoadIntegration: context => {
      return dataStreamDetailsClient.checkAndLoadIntegration({
        dataStream: context.dataStream
      });
    },
    loadIntegrationDashboards: context => {
      if ('integration' in context && context.integration && context.integration.integration) {
        return dataStreamDetailsClient.getIntegrationDashboards({
          integration: context.integration.integration.name
        });
      }
      return Promise.resolve();
    },
    saveNewFieldLimit: context => {
      if ('fieldLimit' in context && context.fieldLimit && context.fieldLimit.newFieldLimit) {
        return dataStreamDetailsClient.setNewFieldLimit({
          dataStream: context.dataStream,
          newFieldLimit: context.fieldLimit.newFieldLimit
        });
      }
      return Promise.resolve();
    },
    rolloverDataStream: context => {
      return dataStreamDetailsClient.rolloverDataStream({
        dataStream: context.dataStream
      });
    }
  }
});
exports.createDatasetQualityDetailsControllerStateMachine = createDatasetQualityDetailsControllerStateMachine;