"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSearchSessionsDeprecationsConfig = void 0;
var _i18n = require("@kbn/i18n");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const createSearchSessionsDeprecationsConfig = core => ({
  getDeprecations: async context => {
    const searchSessionsLink = core.http.basePath.prepend('/app/management/kibana/search_sessions');
    const [coreStart] = await core.getStartServices();
    const savedObjectsClient = coreStart.savedObjects.getScopedClient(context.request, {
      includedHiddenTypes: [_common.SEARCH_SESSION_TYPE]
    });
    const results = await savedObjectsClient.find({
      type: 'search-session',
      perPage: 1000,
      fields: ['name', 'username', 'expires'],
      sortField: 'created',
      sortOrder: 'desc',
      namespaces: ['*']
    });
    const searchSessions = results.saved_objects.filter(so => new Date(so.attributes.expires).getTime() > Date.now());
    if (!searchSessions.length) {
      return [];
    }
    return [{
      title: _i18n.i18n.translate('data.deprecations.searchSessionsTitle', {
        defaultMessage: 'Search sessions will be disabled by default'
      }),
      message: buildMessage({
        searchSessions,
        searchSessionsLink
      }),
      deprecationType: 'feature',
      level: 'warning',
      correctiveActions: {
        manualSteps: [_i18n.i18n.translate('data.deprecations.searchSessions.manualStepOneMessage', {
          defaultMessage: 'Navigate to Stack Management > Kibana > Search Sessions'
        }), _i18n.i18n.translate('data.deprecations.searchSessions.manualStepTwoMessage', {
          defaultMessage: 'Delete search sessions that have not expired'
        }), _i18n.i18n.translate('data.deprecations.searchSessions.manualStepTwoMessage', {
          defaultMessage: 'Alternatively, to continue using search sessions until 9.1, open the kibana.yml config file and add the following: "data.search.sessions.enabled: true"'
        })]
      }
    }];
  }
});
exports.createSearchSessionsDeprecationsConfig = createSearchSessionsDeprecationsConfig;
const buildMessage = ({
  searchSessions,
  searchSessionsLink
}) => ({
  type: 'markdown',
  content: _i18n.i18n.translate('data.deprecations.scriptedFieldsMessage', {
    defaultMessage: `The search sessions feature is deprecated and is disabled by default in 9.0. You currently have {numberOfSearchSessions} active search session(s): [Manage Search Sessions]({searchSessionsLink})`,
    values: {
      numberOfSearchSessions: searchSessions.length,
      searchSessionsLink
    }
  })
});