"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createFiltersFromValueClickAction = exports.createFilter = exports.appendFilterToESQLQueryFromValueClickAction = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _esQuery = require("@kbn/es-query");
var _esqlUtils = require("@kbn/esql-utils");
var _services = require("../../services");
var _query = require("../../query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * For terms aggregations on `__other__` buckets, this assembles a list of applicable filter
 * terms based on a specific cell in the tabified data.
 *
 * @param  {EventData['table']} table - tabified table data
 * @param  {number} columnIndex - current column index
 * @param  {number} rowIndex - current row index
 * @return {array} - array of terms to filter against
 */
const getOtherBucketFilterTerms = (table, columnIndex, rowIndex) => {
  if (rowIndex === -1) {
    return [];
  }

  // get only rows where cell value matches current row for all the fields before columnIndex
  const rows = table.rows.filter(row => {
    return table.columns.every((column, i) => {
      return row[column.id] === table.rows[rowIndex][column.id] || i >= columnIndex;
    });
  });
  const terms = rows.map(row => row[table.columns[columnIndex].id]);
  return [...new Set(terms.filter(term => {
    const notOther = String(term) !== '__other__';
    const notMissing = String(term) !== '__missing__';
    return notOther && notMissing;
  }))];
};

/**
 * Assembles the filters needed to apply filtering against a specific cell value, while accounting
 * for cases like if the value is a terms agg in an `__other__` or `__missing__` bucket.
 *
 * @param  {EventData['table']} table - tabified table data
 * @param  {number} columnIndex - current column index
 * @param  {number} rowIndex - current row index
 * @param  {string} cellValue - value of the current cell
 * @return {Filter[]|undefined} - list of filters to provide to queryFilter.addFilters()
 */
const createFilter = async (table, columnIndex, rowIndex) => {
  var _table$columns$column;
  if (!table || !table.columns || !table.columns[columnIndex] || !table.columns[columnIndex].meta || table.columns[columnIndex].meta.source !== 'esaggs' || !((_table$columns$column = table.columns[columnIndex].meta.sourceParams) !== null && _table$columns$column !== void 0 && _table$columns$column.indexPatternId)) {
    return;
  }
  const column = table.columns[columnIndex];
  const {
    indexPatternId,
    ...aggConfigParams
  } = table.columns[columnIndex].meta.sourceParams;
  const aggConfigsInstance = (0, _services.getSearchService)().aggs.createAggConfigs(await (0, _services.getIndexPatterns)().get(indexPatternId), [aggConfigParams]);
  const aggConfig = aggConfigsInstance.aggs[0];
  let filter = [];
  const value = rowIndex > -1 ? table.rows[rowIndex][column.id] : null;
  if (value === null || value === undefined || !aggConfig.isFilterable()) {
    return;
  }
  if ((aggConfig.type.name === 'terms' || aggConfig.type.name === 'multi_terms') && aggConfig.params.otherBucket) {
    const terms = getOtherBucketFilterTerms(table, columnIndex, rowIndex);
    filter = aggConfig.createFilter(value, {
      terms
    });
  } else {
    filter = aggConfig.createFilter(value);
  }
  if (!filter) {
    return;
  }
  if (!Array.isArray(filter)) {
    filter = [filter];
  }
  return filter;
};

/** @public */
exports.createFilter = createFilter;
const createFiltersFromValueClickAction = async ({
  data,
  negate
}) => {
  const filters = [];
  await Promise.all(data.filter(point => point).map(async val => {
    const {
      table,
      column,
      row
    } = val;
    const filter = (await createFilter(table, column, row)) || [];
    if (filter) {
      filter.forEach(f => {
        if (negate) {
          f = (0, _esQuery.toggleFilterNegated)(f);
        }
        filters.push(f);
      });
    }
  }));
  return _lodash.default.uniqWith((0, _query.mapAndFlattenFilters)(filters), (a, b) => (0, _esQuery.compareFilters)(a, b, _esQuery.COMPARE_ALL_OPTIONS));
};

/** @public */
exports.createFiltersFromValueClickAction = createFiltersFromValueClickAction;
const appendFilterToESQLQueryFromValueClickAction = ({
  data,
  query,
  negate
}) => {
  var _table$columns;
  if (!query) {
    return;
  }
  // Do not append in case of time series, for now. We need to find a way to compute the interval
  // to create the time range filter correctly. The users can brush to update the time filter instead.
  const dataPoints = data.filter(point => {
    var _point$table, _point$table$columns, _point$table$columns$, _point$table$columns$2;
    return point && ((_point$table = point.table) === null || _point$table === void 0 ? void 0 : (_point$table$columns = _point$table.columns) === null || _point$table$columns === void 0 ? void 0 : (_point$table$columns$ = _point$table$columns[point.column]) === null || _point$table$columns$ === void 0 ? void 0 : (_point$table$columns$2 = _point$table$columns$.meta) === null || _point$table$columns$2 === void 0 ? void 0 : _point$table$columns$2.type) !== 'date';
  });
  if (!dataPoints.length) {
    return;
  }
  const {
    table,
    column: columnIndex,
    row: rowIndex
  } = dataPoints[dataPoints.length - 1];
  if (table !== null && table !== void 0 && (_table$columns = table.columns) !== null && _table$columns !== void 0 && _table$columns[columnIndex]) {
    var _column$meta;
    const column = table.columns[columnIndex];
    const value = rowIndex > -1 ? table.rows[rowIndex][column.id] : null;
    if (value == null) {
      return;
    }
    return (0, _esqlUtils.appendWhereClauseToESQLQuery)(query.esql, column.name, value, negate ? '-' : '+', (_column$meta = column.meta) === null || _column$meta === void 0 ? void 0 : _column$meta.type);
  }
};
exports.appendFilterToESQLQueryFromValueClickAction = appendFilterToESQLQueryFromValueClickAction;