"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.openSaveModal = openSaveModal;
var _react = _interopRequireDefault(require("react"));
var _ebtTools = require("@kbn/ebt-tools");
var _public = require("@kbn/saved-objects-plugin/public");
var _i18n = require("@kbn/i18n");
var _kibana_services = require("../services/kibana_services");
var _dashboard_content_management_service = require("../services/dashboard_content_management_service");
var _telemetry_constants = require("../utils/telemetry_constants");
var _extract_title_and_count = require("../dashboard_container/embeddable/api/lib/extract_title_and_count");
var _save_modal = require("../dashboard_container/embeddable/api/overlays/save_modal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * @description exclusively for user directed dashboard save actions, also
 * accounts for scenarios of cloning elastic managed dashboard into user managed dashboards
 */
async function openSaveModal({
  controlGroupReferences,
  dashboardState,
  isManaged,
  lastSavedId,
  panelReferences,
  searchSourceReferences,
  viewMode
}) {
  if (viewMode === 'edit' && isManaged) {
    return undefined;
  }
  const dashboardContentManagementService = (0, _dashboard_content_management_service.getDashboardContentManagementService)();
  const saveAsTitle = lastSavedId ? await getSaveAsTitle(dashboardState.title) : dashboardState.title;
  return new Promise((resolve, reject) => {
    var _dashboardState$descr;
    const onSaveAttempt = async ({
      newTags,
      newTitle,
      newDescription,
      newCopyOnSave,
      newTimeRestore,
      onTitleDuplicate,
      isTitleDuplicateConfirmed
    }) => {
      const saveOptions = {
        confirmOverwrite: false,
        isTitleDuplicateConfirmed,
        onTitleDuplicate,
        saveAsCopy: lastSavedId ? true : newCopyOnSave
      };
      try {
        if (!(await dashboardContentManagementService.checkForDuplicateDashboardTitle({
          title: newTitle,
          onTitleDuplicate,
          lastSavedTitle: dashboardState.title,
          copyOnSave: saveOptions.saveAsCopy,
          isTitleDuplicateConfirmed
        }))) {
          return {};
        }
        const dashboardStateToSave = {
          ...dashboardState,
          title: newTitle,
          tags: _kibana_services.savedObjectsTaggingService && newTags ? newTags : [],
          description: newDescription,
          timeRestore: newTimeRestore,
          timeRange: newTimeRestore ? _kibana_services.dataService.query.timefilter.timefilter.getTime() : undefined,
          refreshInterval: newTimeRestore ? _kibana_services.dataService.query.timefilter.timefilter.getRefreshInterval() : undefined
        };

        // TODO If this is a managed dashboard - unlink all by reference embeddables on clone
        // https://github.com/elastic/kibana/issues/190138

        const beforeAddTime = window.performance.now();
        const saveResult = await dashboardContentManagementService.saveDashboardState({
          controlGroupReferences,
          panelReferences,
          searchSourceReferences,
          saveOptions,
          dashboardState: dashboardStateToSave,
          lastSavedId
        });
        const addDuration = window.performance.now() - beforeAddTime;
        (0, _ebtTools.reportPerformanceMetricEvent)(_kibana_services.coreServices.analytics, {
          eventName: _telemetry_constants.SAVED_OBJECT_POST_TIME,
          duration: addDuration,
          meta: {
            saved_object_type: _telemetry_constants.DASHBOARD_CONTENT_ID
          }
        });
        resolve({
          ...saveResult,
          savedState: dashboardStateToSave
        });
        return saveResult;
      } catch (error) {
        reject(error);
        return error;
      }
    };
    (0, _public.showSaveModal)(/*#__PURE__*/_react.default.createElement(_save_modal.DashboardSaveModal, {
      tags: dashboardState.tags,
      title: saveAsTitle,
      onClose: () => resolve(undefined),
      timeRestore: dashboardState.timeRestore,
      showStoreTimeOnSave: !lastSavedId,
      description: (_dashboardState$descr = dashboardState.description) !== null && _dashboardState$descr !== void 0 ? _dashboardState$descr : '',
      showCopyOnSave: false,
      onSave: onSaveAttempt,
      customModalTitle: getCustomModalTitle(viewMode)
    }));
  });
}
function getCustomModalTitle(viewMode) {
  if (viewMode === 'edit') return _i18n.i18n.translate('dashboard.topNav.editModeInteractiveSave.modalTitle', {
    defaultMessage: 'Save as new dashboard'
  });
  if (viewMode === 'view') return _i18n.i18n.translate('dashboard.topNav.viewModeInteractiveSave.modalTitle', {
    defaultMessage: 'Duplicate dashboard'
  });
  return undefined;
}
async function getSaveAsTitle(title) {
  const [baseTitle, baseCount] = (0, _extract_title_and_count.extractTitleAndCount)(title);
  let saveAsTitle = `${baseTitle} (${baseCount + 1})`;
  await (0, _dashboard_content_management_service.getDashboardContentManagementService)().checkForDuplicateDashboardTitle({
    title: saveAsTitle,
    lastSavedTitle: title,
    copyOnSave: true,
    isTitleDuplicateConfirmed: false,
    onTitleDuplicate(speculativeSuggestion) {
      saveAsTitle = speculativeSuggestion;
    }
  });
  return saveAsTitle;
}