"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FeatureFlagsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _apmRum = require("@elastic/apm-rum");
var _webSdk = require("@openfeature/web-sdk");
var _deepmerge = _interopRequireDefault(require("deepmerge"));
var _rxjs = require("rxjs");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * setup method dependencies
 * @private
 */

/**
 * The browser-side Feature Flags Service
 * @private
 */
class FeatureFlagsService {
  /**
   * The core service's constructor
   * @param core {@link CoreContext}
   */
  constructor(core) {
    (0, _defineProperty2.default)(this, "featureFlagsClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "isProviderReadyPromise", void 0);
    (0, _defineProperty2.default)(this, "context", {
      kind: 'multi'
    });
    (0, _defineProperty2.default)(this, "overrides", {});
    this.logger = core.logger.get('feature-flags-service');
    this.featureFlagsClient = _webSdk.OpenFeature.getClient();
    _webSdk.OpenFeature.setLogger(this.logger.get('open-feature'));
  }

  /**
   * Setup lifecycle method
   * @param deps {@link FeatureFlagsSetup} including the {@link InternalInjectedMetadataSetup} used to retrieve the feature flags.
   */
  setup(deps) {
    const featureFlagsInjectedMetadata = deps.injectedMetadata.getFeatureFlags();
    if (featureFlagsInjectedMetadata) {
      this.overrides = featureFlagsInjectedMetadata.overrides;
    }
    return {
      setProvider: provider => {
        if (this.isProviderReadyPromise) {
          throw new Error('A provider has already been set. This API cannot be called twice.');
        }
        const transaction = _apmRum.apm.startTransaction('set-provider', 'feature-flags');
        this.isProviderReadyPromise = _webSdk.OpenFeature.setProviderAndWait(provider);
        this.isProviderReadyPromise.then(() => {
          if (transaction) {
            // @ts-expect-error RUM types are not correct
            transaction.outcome = 'success';
            transaction.end();
          }
        }).catch(err => {
          this.logger.error(err);
          _apmRum.apm.captureError(err);
          if (transaction) {
            // @ts-expect-error RUM types are not correct
            transaction.outcome = 'failure';
            transaction.end();
          }
        });
      },
      appendContext: contextToAppend => this.appendContext(contextToAppend)
    };
  }

  /**
   * Start lifecycle method
   */
  async start() {
    const featureFlagsChanged$ = new _rxjs.Subject();
    this.featureFlagsClient.addHandler(_webSdk.ClientProviderEvents.ConfigurationChanged, event => {
      if (event !== null && event !== void 0 && event.flagsChanged) {
        featureFlagsChanged$.next(event.flagsChanged);
      }
    });
    const observeFeatureFlag$ = flagName => featureFlagsChanged$.pipe((0, _rxjs.filter)(flagNames => flagNames.includes(flagName)), (0, _rxjs.startWith)([flagName]) // only to emit on the first call
    );
    await this.waitForProviderInitialization();
    return {
      appendContext: contextToAppend => this.appendContext(contextToAppend),
      getBooleanValue: (flagName, fallbackValue) => this.evaluateFlag(this.featureFlagsClient.getBooleanValue, flagName, fallbackValue),
      getStringValue: (flagName, fallbackValue) => this.evaluateFlag(this.featureFlagsClient.getStringValue, flagName, fallbackValue),
      getNumberValue: (flagName, fallbackValue) => this.evaluateFlag(this.featureFlagsClient.getNumberValue, flagName, fallbackValue),
      getBooleanValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.map)(() => this.evaluateFlag(this.featureFlagsClient.getBooleanValue, flagName, fallbackValue)));
      },
      getStringValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.map)(() => this.evaluateFlag(this.featureFlagsClient.getStringValue, flagName, fallbackValue)));
      },
      getNumberValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.map)(() => this.evaluateFlag(this.featureFlagsClient.getNumberValue, flagName, fallbackValue)));
      }
    };
  }

  /**
   * Stop lifecycle method
   */
  async stop() {
    await _webSdk.OpenFeature.close();
  }

  /**
   * Waits for the provider initialization with a timeout to avoid holding the page load for too long
   * @private
   */
  async waitForProviderInitialization() {
    // Adding a timeout here to avoid hanging the start for too long if the provider is unresponsive
    let timeoutId;
    await Promise.race([this.isProviderReadyPromise, new Promise(resolve => {
      timeoutId = setTimeout(resolve, 2 * 1000);
    }).then(() => {
      const msg = `The feature flags provider took too long to initialize.
        Won't hold the page load any longer.
        Feature flags will return the provided fallbacks until the provider is eventually initialized.`;
      this.logger.warn(msg);
      _apmRum.apm.captureError(msg);
    })]);
    clearTimeout(timeoutId);
  }

  /**
   * Wrapper to evaluate flags with the common config overrides interceptions + APM and counters reporting
   * @param evaluationFn The actual evaluation API
   * @param flagName The name of the flag to evaluate
   * @param fallbackValue The fallback value
   * @private
   */
  evaluateFlag(evaluationFn, flagName, fallbackValue) {
    const override = (0, _lodash.get)(this.overrides, flagName); // using lodash get because flagName can come with dots and the config parser might structure it in objects.
    const value = typeof override !== 'undefined' ? override :
    // We have to bind the evaluation or the client will lose its internal context
    evaluationFn.bind(this.featureFlagsClient)(flagName, fallbackValue);
    _apmRum.apm.addLabels({
      [`flag_${flagName}`]: value
    });
    // TODO: increment usage counter
    return value;
  }

  /**
   * Formats the provided context to fulfill the expected multi-context structure.
   * @param contextToAppend The {@link EvaluationContext} to append.
   * @private
   */
  async appendContext(contextToAppend) {
    // If no kind provided, default to the project|deployment level.
    const {
      kind = 'kibana',
      ...rest
    } = contextToAppend;
    // Format the context to fulfill the expected multi-context structure
    const formattedContextToAppend = kind === 'multi' ? contextToAppend : {
      kind: 'multi',
      [kind]: rest
    };

    // Merge the formatted context to append to the global context, and set it in the OpenFeature client.
    this.context = (0, _deepmerge.default)(this.context, formattedContextToAppend);
    await _webSdk.OpenFeature.setContext(this.context);
  }
}
exports.FeatureFlagsService = FeatureFlagsService;