"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateTypeMigrations = validateTypeMigrations;
var _semver = _interopRequireDefault(require("semver"));
var _std = require("@kbn/std");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Validates the consistency of the given type for use with the document migrator.
 */
function validateTypeMigrations({
  type,
  kibanaVersion,
  convertVersion
}) {
  if (type.migrations) {
    var _type$migrations;
    assertObjectOrFunction(type.migrations, `Migration for type ${type.name} should be an object or a function returning an object like { '2.0.0': (doc) => doc }.`);
    const migrationMap = typeof type.migrations === 'function' ? type.migrations() : (_type$migrations = type.migrations) !== null && _type$migrations !== void 0 ? _type$migrations : {};
    assertObject(migrationMap, `Migrations map for type ${type.name} should be an object like { '2.0.0': (doc) => doc }.`);
    Object.entries(migrationMap).forEach(([version, migration]) => {
      assertValidSemver(kibanaVersion, version, type.name);
      assertValidTransform(migration, version, type.name);
      if (_semver.default.gte(version, _coreSavedObjectsBaseServerInternal.globalSwitchToModelVersionAt)) {
        throw new Error(`Migration for type ${type.name} for version ${version} registered after globalSwitchToModelVersionAt (${_coreSavedObjectsBaseServerInternal.globalSwitchToModelVersionAt})`);
      }
    });
  }
  if (type.schemas) {
    const schemaMap = typeof type.schemas === 'object' ? type.schemas : {};
    assertObject(schemaMap, `Schemas map for type ${type.name} should be an object like { '2.0.0': {schema} }.`);
    Object.entries(schemaMap).forEach(([version, schema]) => {
      assertValidSemver(kibanaVersion, version, type.name);
      if (_semver.default.gte(version, _coreSavedObjectsBaseServerInternal.globalSwitchToModelVersionAt)) {
        throw new Error(`Schema for type ${type.name} for version ${version} registered after globalSwitchToModelVersionAt (${_coreSavedObjectsBaseServerInternal.globalSwitchToModelVersionAt})`);
      }
    });
  }
  if (type.modelVersions) {
    var _type$modelVersions;
    const modelVersionMap = typeof type.modelVersions === 'function' ? type.modelVersions() : (_type$modelVersions = type.modelVersions) !== null && _type$modelVersions !== void 0 ? _type$modelVersions : {};
    if (Object.keys(modelVersionMap).length > 0) {
      if (!_coreSavedObjectsBaseServerInternal.globalSwitchToModelVersionAt) {
        throw new Error(`Type ${type.name}: Using modelVersions requires to specify globalSwitchToModelVersionAt`);
      }
      Object.entries(modelVersionMap).forEach(([version, definition]) => {
        (0, _coreSavedObjectsBaseServerInternal.assertValidModelVersion)(version);
      });
      const {
        min: minVersion,
        max: maxVersion
      } = Object.keys(modelVersionMap).reduce((minMax, rawVersion) => {
        const version = Number.parseInt(rawVersion, 10);
        minMax.min = Math.min(minMax.min, version);
        minMax.max = Math.max(minMax.max, version);
        return minMax;
      }, {
        min: Infinity,
        max: -Infinity
      });
      if (minVersion > 1) {
        throw new Error(`Type ${type.name}: model versioning must start with version 1`);
      }
      validateAddedMappings(type.name, type.mappings, modelVersionMap);
      const missingVersions = getMissingVersions(minVersion, maxVersion, Object.keys(modelVersionMap).map(v => Number.parseInt(v, 10)));
      if (missingVersions.length) {
        throw new Error(`Type ${type.name}: gaps between model versions aren't allowed (missing versions: ${missingVersions.join(',')})`);
      }
    }
  }
  if (type.convertToMultiNamespaceTypeVersion) {
    assertValidConvertToMultiNamespaceType(kibanaVersion, convertVersion, type.namespaceType, type.convertToMultiNamespaceTypeVersion, type.name);
  }
}
function isMappingAddition(change) {
  return change.type === 'mappings_addition';
}
const validateAddedMappings = (typeName, mappings, modelVersions) => {
  const flattenedMappings = new Map(Object.entries((0, _std.getFlattenedObject)(mappings.properties)));
  const mappingAdditionChanges = Object.values(modelVersions).flatMap(version => version.changes).filter(isMappingAddition);
  const addedMappings = mappingAdditionChanges.reduce((map, change) => {
    const flattened = (0, _std.getFlattenedObject)(change.addedMappings);
    Object.keys(flattened).forEach(key => {
      map.set(key, flattened[key]);
    });
    return map;
  }, new Map());
  const missingMappings = [];
  const mappingsWithDifferentValues = [];
  for (const [key, value] of addedMappings.entries()) {
    if (!flattenedMappings.has(key)) {
      missingMappings.push(key);
    } else {
      const valueInMappings = flattenedMappings.get(key);
      if (valueInMappings !== value) {
        mappingsWithDifferentValues.push(key);
      }
    }
  }
  if (missingMappings.length) {
    throw new Error(`Type ${typeName}: mappings added on model versions not present on the global mappings definition: ${missingMappings.join(',')}`);
  }
  if (mappingsWithDifferentValues.length) {
    throw new Error(`Type ${typeName}: mappings added on model versions differs from the global mappings definition: ${mappingsWithDifferentValues.join(',')}`);
  }
};
const assertObjectOrFunction = (entity, prefix) => {
  if (!entity || typeof entity !== 'function' && typeof entity !== 'object') {
    throw new Error(`${prefix} Got! ${typeof entity}, ${JSON.stringify(entity)}.`);
  }
};
const assertObject = (obj, prefix) => {
  if (!obj || typeof obj !== 'object') {
    throw new Error(`${prefix} Got ${obj}.`);
  }
};
const assertValidSemver = (kibanaVersion, version, type) => {
  if (!_semver.default.valid(version)) {
    throw new Error(`Invalid migration for type ${type}. Expected all properties to be semvers, but got ${version}.`);
  }
  if (_semver.default.gt(version, kibanaVersion)) {
    throw new Error(`Invalid migration for type ${type}. Property '${version}' cannot be greater than the current Kibana version '${kibanaVersion}'.`);
  }
};
const assertValidConvertToMultiNamespaceType = (kibanaVersion, convertVersion, namespaceType, convertToMultiNamespaceTypeVersion, type) => {
  if (namespaceType !== 'multiple' && namespaceType !== 'multiple-isolated') {
    throw new Error(`Invalid convertToMultiNamespaceTypeVersion for type ${type}. Expected namespaceType to be 'multiple' or 'multiple-isolated', but got '${namespaceType}'.`);
  } else if (!_semver.default.valid(convertToMultiNamespaceTypeVersion)) {
    throw new Error(`Invalid convertToMultiNamespaceTypeVersion for type ${type}. Expected value to be a semver, but got '${convertToMultiNamespaceTypeVersion}'.`);
  } else if (convertVersion && _semver.default.neq(convertToMultiNamespaceTypeVersion, convertVersion)) {
    throw new Error(`Invalid convertToMultiNamespaceTypeVersion for type ${type}. Value '${convertToMultiNamespaceTypeVersion}' cannot be any other than '${convertVersion}'.`);
  } else if (_semver.default.gt(convertToMultiNamespaceTypeVersion, kibanaVersion)) {
    throw new Error(`Invalid convertToMultiNamespaceTypeVersion for type ${type}. Value '${convertToMultiNamespaceTypeVersion}' cannot be greater than the current Kibana version '${kibanaVersion}'.`);
  } else if (_semver.default.patch(convertToMultiNamespaceTypeVersion)) {
    throw new Error(`Invalid convertToMultiNamespaceTypeVersion for type ${type}. Value '${convertToMultiNamespaceTypeVersion}' cannot be used on a patch version (must be like 'x.y.0').`);
  }
};
const assertValidTransform = (migration, version, type) => {
  if (!(typeof migration === 'object' && typeof migration.transform === 'function') && typeof migration !== 'function') {
    throw new Error(`Invalid migration ${type}.${version}: expected a function or an object, but got ${migration}.`);
  }
};
const getMissingVersions = (from, to, versions) => {
  const versionSet = new Set(versions);
  const missing = [];
  for (let i = from; i <= to; i++) {
    if (!versionSet.has(i)) {
      missing.push(i);
    }
  }
  return missing;
};