"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performCreate = void 0;
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _constants = require("../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const performCreate = async ({
  type,
  attributes,
  options
}, {
  registry,
  helpers,
  allowedTypes,
  client,
  serializer,
  migrator,
  extensions = {}
}) => {
  var _preflightResult$exis, _preflightResult3, _preflightResult3$exi, _preflightResult3$exi2, _preflightResult4;
  const {
    common: commonHelper,
    validation: validationHelper,
    encryption: encryptionHelper,
    preflight: preflightHelper,
    serializer: serializerHelper,
    migration: migrationHelper,
    user: userHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  const {
    migrationVersion,
    coreMigrationVersion,
    typeMigrationVersion,
    managed,
    overwrite = false,
    references = [],
    refresh = _constants.DEFAULT_REFRESH_SETTING,
    initialNamespaces,
    version
  } = options;
  const {
    migrationVersionCompatibility
  } = options;
  if (!allowedTypes.includes(type)) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type);
  }
  const id = commonHelper.getValidId(type, options.id, options.version, options.overwrite);
  validationHelper.validateInitialNamespaces(type, initialNamespaces);
  validationHelper.validateOriginId(type, options);
  const time = (0, _utils.getCurrentTime)();
  const createdBy = userHelper.getCurrentUserProfileUid();
  const updatedBy = createdBy;
  let savedObjectNamespace;
  let savedObjectNamespaces;
  let existingOriginId;
  const namespaceString = _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace);
  let preflightResult;
  if (registry.isSingleNamespace(type)) {
    savedObjectNamespace = initialNamespaces ? (0, _utils.normalizeNamespace)(initialNamespaces[0]) : namespace;
  } else if (registry.isMultiNamespace(type)) {
    var _preflightResult, _preflightResult2, _preflightResult2$exi, _preflightResult2$exi2;
    if (options.id) {
      // we will overwrite a multi-namespace saved object if it exists; if that happens, ensure we preserve its included namespaces
      // note: this check throws an error if the object is found but does not exist in this namespace
      preflightResult = (await preflightHelper.preflightCheckForCreate([{
        type,
        id,
        overwrite,
        namespaces: initialNamespaces !== null && initialNamespaces !== void 0 ? initialNamespaces : [namespaceString]
      }]))[0];
    }
    savedObjectNamespaces = initialNamespaces || (0, _utils.getSavedObjectNamespaces)(namespace, (_preflightResult = preflightResult) === null || _preflightResult === void 0 ? void 0 : _preflightResult.existingDocument);
    existingOriginId = (_preflightResult2 = preflightResult) === null || _preflightResult2 === void 0 ? void 0 : (_preflightResult2$exi = _preflightResult2.existingDocument) === null || _preflightResult2$exi === void 0 ? void 0 : (_preflightResult2$exi2 = _preflightResult2$exi._source) === null || _preflightResult2$exi2 === void 0 ? void 0 : _preflightResult2$exi2.originId;
  }
  const authorizationResult = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeCreate({
    namespace,
    object: {
      type,
      id,
      initialNamespaces,
      existingNamespaces: (_preflightResult$exis = (_preflightResult3 = preflightResult) === null || _preflightResult3 === void 0 ? void 0 : (_preflightResult3$exi = _preflightResult3.existingDocument) === null || _preflightResult3$exi === void 0 ? void 0 : (_preflightResult3$exi2 = _preflightResult3$exi._source) === null || _preflightResult3$exi2 === void 0 ? void 0 : _preflightResult3$exi2.namespaces) !== null && _preflightResult$exis !== void 0 ? _preflightResult$exis : []
    }
  }));
  if ((_preflightResult4 = preflightResult) !== null && _preflightResult4 !== void 0 && _preflightResult4.error) {
    // This intentionally occurs _after_ the authZ enforcement (which may throw a 403 error earlier)
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id);
  }

  // 1. If the originId has been *explicitly set* in the options (defined or undefined), respect that.
  // 2. Otherwise, preserve the originId of the existing object that is being overwritten, if any.
  const originId = Object.keys(options).includes('originId') ? options.originId : existingOriginId;
  const migrated = migrationHelper.migrateInputDocument({
    id,
    type,
    ...(savedObjectNamespace && {
      namespace: savedObjectNamespace
    }),
    ...(savedObjectNamespaces && {
      namespaces: savedObjectNamespaces
    }),
    originId,
    attributes: await encryptionHelper.optionallyEncryptAttributes(type, id, savedObjectNamespace,
    // if single namespace type, this is the first in initialNamespaces. If multi-namespace type this is options.namespace/current namespace.
    attributes),
    migrationVersion,
    coreMigrationVersion,
    typeMigrationVersion,
    managed: (0, _utils.setManaged)({
      optionsManaged: managed
    }),
    created_at: time,
    updated_at: time,
    ...(createdBy && {
      created_by: createdBy
    }),
    ...(updatedBy && {
      updated_by: updatedBy
    }),
    ...(Array.isArray(references) && {
      references
    })
  });

  /**
   * If a validation has been registered for this type, we run it against the migrated attributes.
   * This is an imperfect solution because malformed attributes could have already caused the
   * migration to fail, but it's the best we can do without devising a way to run validations
   * inside the migration algorithm itself.
   */
  validationHelper.validateObjectForCreate(type, migrated);
  const raw = serializer.savedObjectToRaw(migrated);
  const requestParams = {
    id: raw._id,
    index: commonHelper.getIndexForType(type),
    refresh,
    document: raw._source,
    ...(overwrite && version ? (0, _coreSavedObjectsBaseServerInternal.decodeRequestVersion)(version) : {}),
    require_alias: true
  };
  const {
    body,
    statusCode,
    headers
  } = id && overwrite ? await client.index(requestParams, {
    meta: true
  }) : await client.create(requestParams, {
    meta: true
  });

  // throw if we can't verify a 404 response is from Elasticsearch
  if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode,
    headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError(id, type);
  }
  return encryptionHelper.optionallyDecryptAndRedactSingleResult(serializerHelper.rawToSavedObject({
    ...raw,
    ...body
  }, {
    migrationVersionCompatibility
  }), authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, attributes);
};
exports.performCreate = performCreate;