"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getControlsInOrder = getControlsInOrder;
exports.getLastUsedDataViewId = getLastUsedDataViewId;
exports.initControlsManager = initControlsManager;
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _lodash = require("lodash");
var _uuid = require("uuid");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _rxjs = require("rxjs");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getControlsInOrder(initialControlPanelsState) {
  return Object.keys(initialControlPanelsState).map(key => ({
    id: key,
    order: initialControlPanelsState[key].order,
    type: initialControlPanelsState[key].type
  })).sort((a, b) => a.order > b.order ? 1 : -1).map(({
    id,
    type
  }) => ({
    id,
    type
  })); // filter out `order`
}
function initControlsManager(
/**
 * Composed from last saved controls state and previous sessions's unsaved changes to controls state
 */
initialControlsState,
/**
 * Observable that publishes last saved controls state only
 */
lastSavedControlsState$) {
  const initialControlIds = Object.keys(initialControlsState);
  const children$ = new _rxjs.BehaviorSubject({});
  let currentControlsState = {
    ...initialControlsState
  };
  const controlsInOrder$ = new _rxjs.BehaviorSubject(getControlsInOrder(initialControlsState));
  const lastUsedDataViewId$ = new _rxjs.BehaviorSubject(getLastUsedDataViewId(controlsInOrder$.value, initialControlsState));
  const lastUsedWidth$ = new _rxjs.BehaviorSubject(_common.DEFAULT_CONTROL_WIDTH);
  const lastUsedGrow$ = new _rxjs.BehaviorSubject(_common.DEFAULT_CONTROL_GROW);
  function untilControlLoaded(id) {
    if (children$.value[id]) {
      return children$.value[id];
    }
    return new Promise(resolve => {
      const subscription = (0, _rxjs.merge)(children$, controlsInOrder$).subscribe(() => {
        if (children$.value[id]) {
          subscription.unsubscribe();
          resolve(children$.value[id]);
          return;
        }

        // control removed before the control finished loading.
        const controlState = controlsInOrder$.value.find(element => element.id === id);
        if (!controlState) {
          subscription.unsubscribe();
          resolve(undefined);
        }
      });
    });
  }
  function getControlApi(controlUuid) {
    return children$.value[controlUuid];
  }
  async function addNewPanel({
    panelType,
    initialState
  }, index) {
    if (initialState !== null && initialState !== void 0 && initialState.dataViewId) {
      lastUsedDataViewId$.next(initialState.dataViewId);
    }
    if (initialState !== null && initialState !== void 0 && initialState.width) {
      lastUsedWidth$.next(initialState.width);
    }
    if (typeof (initialState === null || initialState === void 0 ? void 0 : initialState.grow) === 'boolean') {
      lastUsedGrow$.next(initialState.grow);
    }
    const id = (0, _uuid.v4)();
    const nextControlsInOrder = [...controlsInOrder$.value];
    nextControlsInOrder.splice(index, 0, {
      id,
      type: panelType
    });
    controlsInOrder$.next(nextControlsInOrder);
    currentControlsState[id] = initialState !== null && initialState !== void 0 ? initialState : {};
    return await untilControlLoaded(id);
  }
  function removePanel(panelId) {
    delete currentControlsState[panelId];
    controlsInOrder$.next(controlsInOrder$.value.filter(({
      id
    }) => id !== panelId));
    children$.next((0, _lodash.omit)(children$.value, panelId));
  }
  return {
    controlsInOrder$,
    getNewControlState: () => {
      return {
        grow: lastUsedGrow$.value,
        width: lastUsedWidth$.value,
        dataViewId: lastUsedDataViewId$.value
      };
    },
    getControlApi,
    setControlApi: (uuid, controlApi) => {
      children$.next({
        ...children$.getValue(),
        [uuid]: controlApi
      });
    },
    serializeControls: () => {
      const references = [];
      const controls = [];
      controlsInOrder$.getValue().forEach(({
        id
      }, index) => {
        const controlApi = getControlApi(id);
        if (!controlApi) {
          return;
        }
        const {
          rawState: {
            grow,
            width,
            ...rest
          },
          references: controlReferences
        } = controlApi.serializeState();
        if (controlReferences && controlReferences.length > 0) {
          references.push(...controlReferences);
        }
        controls.push({
          grow,
          order: index,
          type: controlApi.type,
          width,
          /** Re-add the `controlConfig` layer on serialize so control group saved object retains shape */
          controlConfig: {
            id,
            ...rest
          }
        });
      });
      return {
        controls,
        references
      };
    },
    snapshotControlsRuntimeState: () => {
      const controlsRuntimeState = {};
      controlsInOrder$.getValue().forEach(({
        id,
        type
      }, index) => {
        const controlApi = getControlApi(id);
        if (controlApi && (0, _presentationPublishing.apiHasSnapshottableState)(controlApi)) {
          controlsRuntimeState[id] = {
            order: index,
            type,
            ...controlApi.snapshotRuntimeState()
          };
        }
      });
      return controlsRuntimeState;
    },
    resetControlsUnsavedChanges: () => {
      currentControlsState = {
        ...lastSavedControlsState$.value
      };
      const nextControlsInOrder = getControlsInOrder(currentControlsState);
      controlsInOrder$.next(nextControlsInOrder);
      const nextControlIds = nextControlsInOrder.map(({
        id
      }) => id);
      const children = {
        ...children$.value
      };
      let modifiedChildren = false;
      Object.keys(children).forEach(controlId => {
        if (!nextControlIds.includes(controlId)) {
          // remove children that no longer exist after reset
          delete children[controlId];
          modifiedChildren = true;
        }
      });
      if (modifiedChildren) {
        children$.next(children);
      }
    },
    api: {
      getSerializedStateForChild: childId => {
        const controlPanelState = currentControlsState[childId];
        return controlPanelState ? {
          rawState: controlPanelState
        } : undefined;
      },
      children$: children$,
      getPanelCount: () => {
        return controlsInOrder$.value.length;
      },
      addNewPanel: async panel => {
        return addNewPanel(panel, controlsInOrder$.value.length);
      },
      removePanel,
      replacePanel: async (panelId, newPanel) => {
        const index = controlsInOrder$.value.findIndex(({
          id
        }) => id === panelId);
        removePanel(panelId);
        const controlApi = await addNewPanel(newPanel, index >= 0 ? index : controlsInOrder$.value.length);
        return controlApi ? controlApi.uuid : '';
      },
      untilInitialized: () => {
        return new Promise(resolve => {
          children$.pipe((0, _rxjs.first)(children => {
            const atLeastOneControlNotInitialized = initialControlIds.some(controlId => !children[controlId]);
            return !atLeastOneControlNotInitialized;
          })).subscribe(() => {
            resolve();
          });
        });
      }
    },
    comparators: {
      controlsInOrder: [controlsInOrder$, next => {},
      // setter does nothing, controlsInOrder$ reset by resetControlsRuntimeState
      _fastDeepEqual.default]
    }
  };
}
function getLastUsedDataViewId(controlsInOrder, initialControlPanelsState) {
  let dataViewId;
  for (let i = controlsInOrder.length - 1; i >= 0; i--) {
    const controlId = controlsInOrder[i].id;
    const controlState = initialControlPanelsState[controlId];
    if (controlState !== null && controlState !== void 0 && controlState.dataViewId) {
      dataViewId = controlState.dataViewId;
      break;
    }
  }
  return dataViewId;
}