"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initRpcRoutes = initRpcRoutes;
var _configSchema = require("@kbn/config-schema");
var _savedObjectsSettings = require("@kbn/saved-objects-settings");
var _msearch = require("../../core/msearch");
var _error_wrapper = require("./error_wrapper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function initRpcRoutes(procedureNames, router, {
  rpc,
  contentRegistry
}) {
  if (procedureNames.length === 0) {
    throw new Error(`No procedure names provided.`);
  }

  /**
   * @apiGroup ContentManagement
   *
   * @api {post} /content_management/rpc/{call} Execute RPC call
   * @apiName RPC
   */
  router.post({
    path: '/api/content_management/rpc/{name}',
    security: {
      authz: {
        enabled: false,
        reason: "This route is opted out from authorization, because it's a wrapper around Saved Object client"
      }
    },
    validate: {
      params: _configSchema.schema.object({
        // @ts-ignore We validate above that procedureNames has at least one item
        // so we can ignore the "Target requires 1 element(s) but source may have fewer." TS error
        name: _configSchema.schema.oneOf(procedureNames.map(fnName => _configSchema.schema.literal(fnName)))
      }),
      // Any object payload can be passed, we will validate the input when calling the rpc handler
      body: _configSchema.schema.maybe(_configSchema.schema.object({}, {
        unknowns: 'allow'
      }))
    }
  }, async (requestHandlerContext, request, response) => {
    try {
      const context = {
        contentRegistry,
        requestHandlerContext,
        request,
        mSearchService: new _msearch.MSearchService({
          getSavedObjectsClient: async () => (await requestHandlerContext.core).savedObjects.client,
          contentRegistry,
          getConfig: {
            listingLimit: async () => (await requestHandlerContext.core).uiSettings.client.get(_savedObjectsSettings.LISTING_LIMIT_SETTING),
            perPage: async () => (await requestHandlerContext.core).uiSettings.client.get(_savedObjectsSettings.PER_PAGE_SETTING)
          }
        })
      };
      const {
        name
      } = request.params;
      const result = await rpc.call(context, name, request.body);
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });
}