"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDependenciesTimeseriesStatistics = getDependenciesTimeseriesStatistics;
exports.parseDependenciesStats = parseDependenciesStats;
var _server = require("@kbn/observability-plugin/server");
var _apm = require("../../../common/es_fields/apm");
var _event_outcome = require("../../../common/event_outcome");
var _get_bucket_size = require("../../../common/utils/get_bucket_size");
var _environment_query = require("../../../common/utils/environment_query");
var _get_offset_in_ms = require("../../../common/utils/get_offset_in_ms");
var _get_is_using_service_destination_metrics = require("../../lib/helpers/spans/get_is_using_service_destination_metrics");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchDependenciesTimeseriesStatistics({
  dependencyNames,
  searchServiceDestinationMetrics,
  apmEventClient,
  start,
  end,
  environment,
  kuery,
  numBuckets
}) {
  var _response$aggregation;
  const response = await apmEventClient.search('get_latency_for_dependency', {
    apm: {
      events: [(0, _get_is_using_service_destination_metrics.getProcessorEventForServiceDestinationStatistics)(searchServiceDestinationMetrics)]
    },
    body: {
      track_total_hits: false,
      size: 0,
      query: {
        bool: {
          filter: [...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery), ...(0, _server.rangeQuery)(start, end), ...(0, _get_is_using_service_destination_metrics.getDocumentTypeFilterForServiceDestinationStatistics)(searchServiceDestinationMetrics), {
            terms: {
              [_apm.SPAN_DESTINATION_SERVICE_RESOURCE]: dependencyNames
            }
          }]
        }
      },
      aggs: {
        dependencies: {
          terms: {
            field: _apm.SPAN_DESTINATION_SERVICE_RESOURCE
          },
          aggs: {
            timeseries: {
              date_histogram: {
                field: '@timestamp',
                fixed_interval: (0, _get_bucket_size.getBucketSize)({
                  start,
                  end,
                  numBuckets,
                  minBucketSize: 60
                }).intervalString,
                extended_bounds: {
                  min: start,
                  max: end
                }
              },
              aggs: {
                // latency
                latency_sum: {
                  sum: {
                    field: (0, _get_is_using_service_destination_metrics.getLatencyFieldForServiceDestinationStatistics)(searchServiceDestinationMetrics)
                  }
                },
                ...(searchServiceDestinationMetrics ? {
                  latency_count: {
                    sum: {
                      field: (0, _get_is_using_service_destination_metrics.getDocCountFieldForServiceDestinationStatistics)(searchServiceDestinationMetrics)
                    }
                  }
                } : {}),
                // error
                ...(searchServiceDestinationMetrics ? {
                  total_count: {
                    sum: {
                      field: (0, _get_is_using_service_destination_metrics.getDocCountFieldForServiceDestinationStatistics)(searchServiceDestinationMetrics)
                    }
                  }
                } : {}),
                failures: {
                  filter: {
                    term: {
                      [_apm.EVENT_OUTCOME]: _event_outcome.EventOutcome.failure
                    }
                  },
                  aggs: {
                    ...(searchServiceDestinationMetrics ? {
                      total_count: {
                        sum: {
                          field: (0, _get_is_using_service_destination_metrics.getDocCountFieldForServiceDestinationStatistics)(searchServiceDestinationMetrics)
                        }
                      }
                    } : {})
                  }
                },
                // throughput
                throughput: {
                  rate: {
                    ...(searchServiceDestinationMetrics ? {
                      field: (0, _get_is_using_service_destination_metrics.getDocCountFieldForServiceDestinationStatistics)(searchServiceDestinationMetrics)
                    } : {}),
                    unit: 'minute'
                  }
                }
              }
            }
          }
        }
      }
    }
  });
  return ((_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : _response$aggregation.dependencies.buckets) || [];
}
function parseDependenciesStats({
  dependencies,
  offsetInMs
}) {
  var _dependencies$reduce;
  return (_dependencies$reduce = dependencies.reduce((acc, bucket) => {
    const stats = {
      latency: [],
      errorRate: [],
      throughput: []
    };
    for (const statsBucket of bucket.timeseries.buckets) {
      var _statsBucket$total_co, _statsBucket$total_co2, _statsBucket$failures, _statsBucket$failures2, _statsBucket$latency_, _statsBucket$latency_2, _statsBucket$latency_3;
      const totalCount = (_statsBucket$total_co = (_statsBucket$total_co2 = statsBucket.total_count) === null || _statsBucket$total_co2 === void 0 ? void 0 : _statsBucket$total_co2.value) !== null && _statsBucket$total_co !== void 0 ? _statsBucket$total_co : statsBucket.doc_count;
      const failureCount = (_statsBucket$failures = (_statsBucket$failures2 = statsBucket.failures.total_count) === null || _statsBucket$failures2 === void 0 ? void 0 : _statsBucket$failures2.value) !== null && _statsBucket$failures !== void 0 ? _statsBucket$failures : statsBucket.failures.doc_count;
      const x = statsBucket.key + offsetInMs;
      stats.latency.push({
        x,
        y: ((_statsBucket$latency_ = statsBucket.latency_sum.value) !== null && _statsBucket$latency_ !== void 0 ? _statsBucket$latency_ : 0) / ((_statsBucket$latency_2 = (_statsBucket$latency_3 = statsBucket.latency_count) === null || _statsBucket$latency_3 === void 0 ? void 0 : _statsBucket$latency_3.value) !== null && _statsBucket$latency_2 !== void 0 ? _statsBucket$latency_2 : statsBucket.doc_count)
      });
      stats.errorRate.push({
        x,
        y: failureCount / totalCount
      });
      stats.throughput.push({
        x,
        y: statsBucket.throughput.value
      });
    }
    acc[bucket.key] = stats;
    return acc;
  }, {})) !== null && _dependencies$reduce !== void 0 ? _dependencies$reduce : {};
}
async function getDependenciesTimeseriesStatistics({
  apmEventClient,
  dependencyNames,
  start,
  end,
  environment,
  kuery,
  searchServiceDestinationMetrics,
  offset,
  numBuckets
}) {
  const {
    offsetInMs,
    startWithOffset,
    endWithOffset
  } = (0, _get_offset_in_ms.getOffsetInMs)({
    start,
    end,
    offset
  });
  const [currentTimeseries, comparisonTimeseries] = await Promise.all([fetchDependenciesTimeseriesStatistics({
    dependencyNames,
    searchServiceDestinationMetrics,
    apmEventClient,
    start,
    end,
    kuery,
    environment,
    numBuckets
  }), offset ? fetchDependenciesTimeseriesStatistics({
    dependencyNames,
    searchServiceDestinationMetrics,
    apmEventClient,
    start: startWithOffset,
    end: endWithOffset,
    kuery,
    environment,
    numBuckets
  }) : null]);
  return {
    currentTimeseries: parseDependenciesStats({
      dependencies: currentTimeseries,
      offsetInMs: 0
    }),
    comparisonTimeseries: comparisonTimeseries !== null && comparisonTimeseries !== void 0 && comparisonTimeseries.length ? parseDependenciesStats({
      dependencies: comparisonTimeseries,
      offsetInMs
    }) : null
  };
}