"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAnomalyDetectionJobs = createAnomalyDetectionJobs;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _uuid = require("uuid");
var _common = require("@kbn/observability-plugin/common");
var _anomaly_detection = require("../../../common/anomaly_detection");
var _apm = require("../../../common/es_fields/apm");
var _environment_query = require("../../../common/utils/environment_query");
var _with_apm_span = require("../../utils/with_apm_span");
var _constants = require("./constants");
var _get_anomaly_detection_jobs = require("./get_anomaly_detection_jobs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_TIMEOUT = '60s';
async function createAnomalyDetectionJobs({
  mlClient,
  esClient,
  indices,
  environments,
  logger,
  esCapabilities
}) {
  if (!mlClient) {
    throw _boom.default.notImplemented(_anomaly_detection.ML_ERRORS.ML_NOT_AVAILABLE);
  }
  const uniqueMlJobEnvs = await getUniqueMlJobEnvs(mlClient, environments, logger);
  if (uniqueMlJobEnvs.length === 0) {
    return [];
  }
  return (0, _with_apm_span.withApmSpan)('create_anomaly_detection_jobs', async () => {
    logger.debug(`Creating ML anomaly detection jobs for environments: [${uniqueMlJobEnvs}].`);
    const apmMetricIndex = indices.metric;
    const responses = [];
    const failedJobs = [];
    if (!esCapabilities.serverless) {
      // Waiting for the index is not enabled in serverless, this could potentially cause
      // problems when creating jobs in parallel
      try {
        await waitForIndexStatus(esClient, apmMetricIndex, 'yellow');
      } catch (err) {
        logger.warn(`Error waiting for ${apmMetricIndex} to turn yellow before creating ML jobs`);
      }
    }

    // Avoid the creation of multiple ml jobs in parallel
    // https://github.com/elastic/elasticsearch/issues/36271
    for (const environment of uniqueMlJobEnvs) {
      try {
        const response = await createAnomalyDetectionJob({
          mlClient,
          environment,
          apmMetricIndex
        });
        if (response.jobs[0].success || !response.jobs[0].error) {
          responses.push(response);
        } else {
          failedJobs.push({
            id: response.jobs[0].id,
            error: response.jobs[0].error
          });
        }
      } catch (e) {
        if (!e.id || !e.error) {
          throw e;
        }
        failedJobs.push({
          id: e.id,
          error: e.error
        });
      }
    }
    const jobResponses = responses.flatMap(response => response.jobs);
    if (failedJobs.length > 0) {
      throw new Error(`An error occurred while creating ML jobs: ${JSON.stringify(failedJobs)}`);
    }
    return jobResponses;
  });
}
async function createAnomalyDetectionJob({
  mlClient,
  environment,
  apmMetricIndex
}) {
  return (0, _with_apm_span.withApmSpan)('create_anomaly_detection_job', async () => {
    const randomToken = (0, _uuid.v4)().substr(-4);
    const anomalyDetectionJob = mlClient.modules.setup({
      moduleId: _constants.ML_MODULE_ID_APM_TRANSACTION,
      prefix: `${_constants.APM_ML_JOB_GROUP}-${(0, _lodash.snakeCase)(environment)}-${randomToken}-`,
      groups: [_constants.APM_ML_JOB_GROUP],
      indexPatternName: apmMetricIndex,
      applyToAllSpaces: true,
      start: (0, _moment.default)().subtract(4, 'weeks').valueOf(),
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.metric
            }
          }, {
            term: {
              [_apm.METRICSET_NAME]: 'transaction'
            }
          }, ...(0, _environment_query.environmentQuery)(environment)]
        }
      },
      startDatafeed: true,
      jobOverrides: [{
        custom_settings: {
          job_tags: {
            environment,
            // identifies this as an APM ML job & facilitates future migrations
            apm_ml_version: 3
          }
        }
      }]
    });
    return anomalyDetectionJob;
  });
}
async function getUniqueMlJobEnvs(mlClient, environments, logger) {
  // skip creation of duplicate ML jobs
  const jobs = await (0, _get_anomaly_detection_jobs.getAnomalyDetectionJobs)(mlClient);
  const existingMlJobEnvs = jobs.filter(job => job.version === 3).map(({
    environment
  }) => environment);
  const requestedExistingMlJobEnvs = environments.filter(env => existingMlJobEnvs.includes(env));
  if (requestedExistingMlJobEnvs.length) {
    logger.warn(`Skipping creation of existing ML jobs for environments: [${requestedExistingMlJobEnvs}]}`);
  }
  return environments.filter(env => !existingMlJobEnvs.includes(env));
}
async function waitForIndexStatus(esClient, index, waitForStatus, timeout = DEFAULT_TIMEOUT) {
  return await esClient.cluster.health({
    index,
    wait_for_status: waitForStatus,
    timeout
  }, {
    retryOnTimeout: true,
    maxRetries: 3
  });
}