"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateGaps = exports.prepareGapForUpdate = void 0;
var _alerting_event_logger = require("../../alerting_event_logger/alerting_event_logger");
var _find_gaps = require("../find_gaps");
var _constants = require("../../../../common/constants");
var _calculate_gaps_state = require("./calculate_gaps_state");
var _update_gap_from_schedule = require("./update_gap_from_schedule");
var _mget_gaps = require("../mget_gaps");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CONFLICT_STATUS_CODE = 409;
const MAX_RETRIES = 3;
const PAGE_SIZE = 500;
const delay = ms => new Promise(resolve => setTimeout(resolve, ms));
const prepareGapForUpdate = async (gap, {
  backfillSchedule,
  savedObjectsRepository,
  shouldRefetchAllBackfills,
  backfillClient,
  actionsClient,
  ruleId
}) => {
  const hasFailedBackfillTask = backfillSchedule === null || backfillSchedule === void 0 ? void 0 : backfillSchedule.some(scheduleItem => scheduleItem.status === _constants.adHocRunStatus.ERROR || scheduleItem.status === _constants.adHocRunStatus.TIMEOUT);
  if (backfillSchedule && !hasFailedBackfillTask) {
    (0, _update_gap_from_schedule.updateGapFromSchedule)({
      gap,
      backfillSchedule
    });
  }
  if (hasFailedBackfillTask || !backfillSchedule || shouldRefetchAllBackfills) {
    await (0, _calculate_gaps_state.calculateGapStateFromAllBackfills)({
      gap,
      savedObjectsRepository,
      ruleId,
      backfillClient,
      actionsClient
    });
  }
  return gap;
};
exports.prepareGapForUpdate = prepareGapForUpdate;
const updateGapBatch = async (gaps, {
  backfillSchedule,
  savedObjectsRepository,
  shouldRefetchAllBackfills,
  backfillClient,
  actionsClient,
  alertingEventLogger,
  logger,
  ruleId,
  eventLogClient,
  retryCount = 0
}) => {
  try {
    // Prepare all gaps for update
    const updatedGaps = [];
    for (const gap of gaps) {
      // we do async request only if there errors in backfill or no backfill schedule
      const updatedGap = await prepareGapForUpdate(gap, {
        backfillSchedule,
        savedObjectsRepository,
        shouldRefetchAllBackfills,
        backfillClient,
        actionsClient,
        ruleId
      });
      updatedGaps.push(updatedGap);
    }

    // Convert gaps to the format expected by updateDocuments
    const gapsToUpdate = updatedGaps.map(gap => {
      if (!gap.internalFields) return null;
      return {
        gap: gap.toObject(),
        internalFields: gap.internalFields
      };
    }).filter(gap => gap !== null);
    if (gapsToUpdate.length === 0) {
      return true;
    }

    // Attempt bulk update
    const bulkResponse = await alertingEventLogger.updateGaps(gapsToUpdate);
    if (bulkResponse.errors) {
      if (retryCount >= MAX_RETRIES) {
        logger.error(`Failed to update ${bulkResponse.items.length} gaps after ${MAX_RETRIES} retries due to conflicts`);
        return false;
      }
      logger.info(`Retrying update of ${bulkResponse.items.length} gaps due to conflicts. Retry ${retryCount + 1} of ${MAX_RETRIES}`);
      const retryDelaySec = Math.min(Math.pow(3, retryCount + 1), 30);
      await delay(retryDelaySec * 1000 * Math.random());
      const failedUpdatesDocs = bulkResponse === null || bulkResponse === void 0 ? void 0 : bulkResponse.items.filter(item => {
        var _item$update;
        return ((_item$update = item.update) === null || _item$update === void 0 ? void 0 : _item$update.status) === CONFLICT_STATUS_CODE;
      }).map(item => {
        var _item$update2, _item$update3;
        return {
          _id: (_item$update2 = item.update) === null || _item$update2 === void 0 ? void 0 : _item$update2._id,
          _index: (_item$update3 = item.update) === null || _item$update3 === void 0 ? void 0 : _item$update3._index
        };
      }).filter(doc => doc._id !== undefined && doc._index !== undefined);

      // Fetch latest versions of failed gaps
      const gapsToRetry = await (0, _mget_gaps.mgetGaps)({
        eventLogClient,
        logger,
        params: {
          docs: failedUpdatesDocs
        }
      });
      if (gapsToRetry.length > 0) {
        // Retry failed gaps
        return updateGapBatch(gapsToRetry, {
          backfillSchedule,
          savedObjectsRepository,
          shouldRefetchAllBackfills,
          backfillClient,
          actionsClient,
          alertingEventLogger,
          logger,
          ruleId,
          eventLogClient,
          retryCount: retryCount + 1
        });
      }
    }
    return true;
  } catch (e) {
    logger.error(`Failed to update gap batch: ${e.message}`);
    return false;
  }
};

/**
 * Update gaps for a given rule
 * Using search_after pagination to process more than 10,000 gaps with stable sorting
 * Prepare gaps for update
 * Update them in bulk
 * If there are conflicts, retry the failed gaps
 */
const updateGaps = async params => {
  const {
    ruleId,
    start,
    end,
    logger,
    eventLogClient,
    eventLogger,
    backfillSchedule,
    savedObjectsRepository,
    shouldRefetchAllBackfills,
    backfillClient,
    actionsClient
  } = params;
  if (!eventLogger) {
    throw new Error('Event logger is required');
  }
  try {
    const alertingEventLogger = new _alerting_event_logger.AlertingEventLogger(eventLogger);
    let hasErrors = false;
    let searchAfter;
    let pitId;
    let iterationCount = 0;
    // Circuit breaker to prevent infinite loops
    // It should be enough to update 50,000,000 gaps
    // 100000 * 500 = 50,000,000 millions gaps
    const MAX_ITERATIONS = 100000;
    try {
      while (true) {
        if (iterationCount >= MAX_ITERATIONS) {
          logger.warn(`Circuit breaker triggered: Reached maximum number of iterations (${MAX_ITERATIONS}) while updating gaps for rule ${ruleId}`);
          break;
        }
        iterationCount++;
        const gapsResponse = await (0, _find_gaps.findGapsSearchAfter)({
          eventLogClient,
          logger,
          params: {
            ruleId,
            start: start.toISOString(),
            end: end.toISOString(),
            perPage: PAGE_SIZE,
            statuses: [_constants.gapStatus.PARTIALLY_FILLED, _constants.gapStatus.UNFILLED],
            sortField: '@timestamp',
            sortOrder: 'asc',
            searchAfter,
            pitId
          }
        });
        const {
          data: gaps,
          searchAfter: nextSearchAfter,
          pitId: nextPitId
        } = gapsResponse;
        pitId = nextPitId;
        if (gaps.length > 0) {
          const success = await updateGapBatch(gaps, {
            backfillSchedule,
            savedObjectsRepository,
            shouldRefetchAllBackfills,
            backfillClient,
            actionsClient,
            alertingEventLogger,
            logger,
            ruleId,
            eventLogClient
          });
          if (!success) {
            hasErrors = true;
          }
        }

        // Exit conditions: no more results or no next search_after
        if (gaps.length === 0 || !nextSearchAfter) {
          break;
        }
        searchAfter = nextSearchAfter;
      }
    } finally {
      if (pitId) {
        await eventLogClient.closePointInTime(pitId);
      }
    }
    if (hasErrors) {
      throw new Error('Some gaps failed to update');
    }
  } catch (e) {
    logger.error(`Failed to update gaps for rule ${ruleId} from: ${start.toISOString()} to: ${end.toISOString()}: ${e.message}`);
  }
};
exports.updateGaps = updateGaps;