"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BackfillClient = exports.BACKFILL_TASK_TYPE = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _server2 = require("@kbn/task-manager-plugin/server");
var _lodash = require("lodash");
var _transforms = require("../application/backfill/transforms");
var _audit_events = require("../rules_client/common/audit_events");
var _saved_objects = require("../saved_objects");
var _lib = require("./lib");
var _update_gaps = require("../lib/rule_gaps/update/update_gaps");
var _denormalize_actions = require("../rules_client/lib/denormalize_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BACKFILL_TASK_TYPE = exports.BACKFILL_TASK_TYPE = 'ad_hoc_run-backfill';
class BackfillClient {
  constructor(opts) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "taskManagerStartPromise", void 0);
    this.logger = opts.logger;
    this.taskManagerStartPromise = opts.taskManagerStartPromise;

    // Registers the task that handles the backfill using the ad hoc task runner
    opts.taskManagerSetup.registerTaskDefinitions({
      [BACKFILL_TASK_TYPE]: {
        title: 'Alerting Backfill Rule Run',
        priority: _server2.TaskPriority.Low,
        createTaskRunner: context => opts.taskRunnerFactory.createAdHoc(context)
      }
    });
  }
  async bulkQueue({
    actionsClient,
    auditLogger,
    params,
    rules,
    ruleTypeRegistry,
    spaceId,
    unsecuredSavedObjectsClient,
    eventLogClient,
    internalSavedObjectsRepository,
    eventLogger
  }) {
    const adHocSOsToCreate = [];

    /**
     * soToCreateIndexOrErrorMap contains a map of the original request index to the
     * AdHocRunSO to create index in the adHocSOsToCreate array or any errors
     * encountered while processing the request
     *
     * For example, if the original request has 5 entries, 2 of which result in errors,
     * the map will look like:
     *
     * params: [request1, request2, request3, request4, request5]
     * adHocSOsToCreate: [AdHocRunSO1, AdHocRunSO3, AdHocRunSO4]
     * soToCreateIndexOrErrorMap: {
     *   0: 0,
     *   1: error1,
     *   2: 1,
     *   3: 2,
     *   4: error2
     * }
     *
     * This allows us to return a response in the same order the requests were received
     */

    const soToCreateIndexOrErrorMap = new Map();
    const rulesWithUnsupportedActions = new Set();
    for (let ndx = 0; ndx < params.length; ndx++) {
      const param = params[ndx];
      // For this schedule request, look up the rule or return error
      const {
        rule,
        error
      } = getRuleOrError({
        ruleId: param.ruleId,
        rules,
        ruleTypeRegistry
      });
      if (rule) {
        // keep track of index of this request in the adHocSOsToCreate array
        soToCreateIndexOrErrorMap.set(ndx, adHocSOsToCreate.length);
        const reference = {
          id: rule.id,
          name: `rule`,
          type: _saved_objects.RULE_SAVED_OBJECT_TYPE
        };
        const {
          actions,
          hasUnsupportedActions,
          references
        } = await extractRuleActions({
          actionsClient,
          rule,
          runActions: param.runActions
        });
        if (hasUnsupportedActions) {
          rulesWithUnsupportedActions.add(ndx);
        }
        adHocSOsToCreate.push({
          type: _saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE,
          attributes: (0, _transforms.transformBackfillParamToAdHocRun)(param, rule, actions, spaceId),
          references: [reference, ...references]
        });
      } else if (error) {
        // keep track of the error encountered for this request by index so
        // we can return it in order
        soToCreateIndexOrErrorMap.set(ndx, error);
        this.logger.warn(`Error for ruleId ${param.ruleId} - not scheduling backfill for ${JSON.stringify(param)}`);
      }
    }

    // Every request encountered an error, so short-circuit the logic here
    if (!adHocSOsToCreate.length) {
      return params.map((_, ndx) => soToCreateIndexOrErrorMap.get(ndx));
    }

    // Bulk create the saved object
    const bulkCreateResponse = await unsecuredSavedObjectsClient.bulkCreate(adHocSOsToCreate);
    const transformedResponse = bulkCreateResponse.saved_objects.map((so, index) => {
      if (so.error) {
        auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.adHocRunAuditEvent)({
          action: _audit_events.AdHocRunAuditAction.CREATE,
          error: new Error(so.error.message)
        }));
      } else {
        auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.adHocRunAuditEvent)({
          action: _audit_events.AdHocRunAuditAction.CREATE,
          savedObject: {
            type: _saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE,
            id: so.id
          }
        }));
      }
      return (0, _transforms.transformAdHocRunToBackfillResult)({
        adHocRunSO: so,
        isSystemAction: id => actionsClient.isSystemAction(id),
        originalSO: adHocSOsToCreate === null || adHocSOsToCreate === void 0 ? void 0 : adHocSOsToCreate[index]
      });
    });

    /**
     * Use soToCreateIndexOrErrorMap to build the result array that returns
     * the bulkQueue result in the same order of the request
     *
     * For example, if we have 3 entries in the bulkCreateResponse
     *
     * bulkCreateResult: [AdHocRunSO1, AdHocRunSO3, AdHocRunSO4]
     * soToCreateIndexOrErrorMap: {
     *   0: 0,
     *   1: error1,
     *   2: 1,
     *   3: 2,
     *   4: error2
     * }
     *
     * The following result would be returned
     * result: [AdHocRunSO1, error1, AdHocRunSO3, AdHocRunSO4, error2]
     */
    const createSOResult = Array.from(soToCreateIndexOrErrorMap.keys()).map(ndx => {
      const indexOrError = soToCreateIndexOrErrorMap.get(ndx);
      if ((0, _lodash.isNumber)(indexOrError)) {
        // This number is the index of the response from the savedObjects bulkCreate function
        const response = transformedResponse[indexOrError];
        if (rulesWithUnsupportedActions.has(indexOrError)) {
          return {
            ...response,
            warnings: [`Rule has actions that are not supported for backfill. Those actions will be skipped.`]
          };
        }
        return response;
      } else {
        // Return the error we encountered
        return indexOrError;
      }
    });

    // Build array of tasks to schedule
    const adHocTasksToSchedule = [];
    const backfillSOs = [];
    createSOResult.forEach(result => {
      if (!result.error) {
        const createdSO = result;
        backfillSOs.push(createdSO);
        const ruleTypeTimeout = ruleTypeRegistry.get(createdSO.rule.alertTypeId).ruleTaskTimeout;
        adHocTasksToSchedule.push({
          id: createdSO.id,
          taskType: BACKFILL_TASK_TYPE,
          ...(ruleTypeTimeout ? {
            timeoutOverride: ruleTypeTimeout
          } : {}),
          state: {},
          params: {
            adHocRunParamsId: createdSO.id,
            spaceId
          }
        });
      }
    });
    try {
      // Process backfills in chunks of 10 to manage resource usage
      for (let i = 0; i < backfillSOs.length; i += 10) {
        const chunk = backfillSOs.slice(i, i + 10);
        await Promise.all(chunk.map(backfill => (0, _update_gaps.updateGaps)({
          backfillSchedule: backfill.schedule,
          ruleId: backfill.rule.id,
          start: new Date(backfill.start),
          end: backfill !== null && backfill !== void 0 && backfill.end ? new Date(backfill.end) : new Date(),
          eventLogger,
          eventLogClient,
          savedObjectsRepository: internalSavedObjectsRepository,
          logger: this.logger,
          backfillClient: this,
          actionsClient
        })));
      }
    } catch {
      this.logger.warn(`Error updating gaps for backfill jobs: ${backfillSOs.map(backfill => backfill.id).join(', ')}`);
    }
    if (adHocTasksToSchedule.length > 0) {
      const taskManager = await this.taskManagerStartPromise;
      await taskManager.bulkSchedule(adHocTasksToSchedule);
    }
    return createSOResult;
  }
  async deleteBackfillForRules({
    ruleIds,
    namespace,
    unsecuredSavedObjectsClient
  }) {
    try {
      // query for all ad hoc runs that reference this ruleId
      const adHocRunFinder = await unsecuredSavedObjectsClient.createPointInTimeFinder({
        type: _saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE,
        perPage: 100,
        hasReference: ruleIds.map(ruleId => ({
          id: ruleId,
          type: _saved_objects.RULE_SAVED_OBJECT_TYPE
        })),
        ...(namespace ? {
          namespaces: [namespace]
        } : undefined)
      });
      const adHocRuns = [];
      for await (const response of adHocRunFinder.find()) {
        adHocRuns.push(...response.saved_objects);
      }
      await adHocRunFinder.close();
      if (adHocRuns.length > 0) {
        const deleteResult = await unsecuredSavedObjectsClient.bulkDelete(adHocRuns.map(adHocRun => ({
          id: adHocRun.id,
          type: _saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE
        })));
        const deleteErrors = deleteResult.statuses.filter(status => !!status.error);
        if (deleteErrors.length > 0) {
          this.logger.warn(`Error deleting backfill jobs with IDs: ${deleteErrors.map(status => status.id).join(', ')} with errors: ${deleteErrors.map(status => {
            var _status$error;
            return (_status$error = status.error) === null || _status$error === void 0 ? void 0 : _status$error.message;
          })} - jobs and associated task were not deleted.`);
        }

        // only delete tasks if the associated ad hoc runs were successfully deleted
        const taskIdsToDelete = deleteResult.statuses.filter(status => status.success).map(status => status.id);

        // delete the associated tasks
        const taskManager = await this.taskManagerStartPromise;
        const deleteTaskResult = await taskManager.bulkRemove(taskIdsToDelete);
        const deleteTaskErrors = deleteTaskResult.statuses.filter(status => !!status.error);
        if (deleteTaskErrors.length > 0) {
          this.logger.warn(`Error deleting tasks with IDs: ${deleteTaskErrors.map(status => status.id).join(', ')} with errors: ${deleteTaskErrors.map(status => {
            var _status$error2;
            return (_status$error2 = status.error) === null || _status$error2 === void 0 ? void 0 : _status$error2.message;
          })}`);
        }
      }
    } catch (error) {
      this.logger.warn(`Error deleting backfill jobs for rule IDs: ${ruleIds.join(',')} - ${error.message}`);
    }
  }
  async findOverlappingBackfills({
    ruleId,
    start,
    end,
    savedObjectsRepository,
    actionsClient
  }) {
    const adHocRuns = [];

    // Create a point in time finder for efficient pagination
    const adHocRunFinder = await savedObjectsRepository.createPointInTimeFinder({
      type: _saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE,
      perPage: 100,
      hasReference: [{
        id: ruleId,
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE
      }],
      filter: `
        ad_hoc_run_params.attributes.start <= "${end.toISOString()}" and
        ad_hoc_run_params.attributes.end >= "${start.toISOString()}"
      `
    });
    try {
      // Collect all results using async iterator
      for await (const response of adHocRunFinder.find()) {
        adHocRuns.push(...response.saved_objects);
      }
    } finally {
      // Make sure we always close the finder
      await adHocRunFinder.close();
    }
    return adHocRuns.map(data => (0, _transforms.transformAdHocRunToBackfillResult)({
      adHocRunSO: data,
      isSystemAction: connectorId => actionsClient.isSystemAction(connectorId)
    }));
  }
}
exports.BackfillClient = BackfillClient;
function getRuleOrError({
  ruleId,
  rules,
  ruleTypeRegistry
}) {
  var _ruleTypeRegistry$get;
  const rule = rules.find(r => r.id === ruleId);

  // if rule not found, return not found error
  if (!rule) {
    const notFoundError = _server.SavedObjectsErrorHelpers.createGenericNotFoundError(_saved_objects.RULE_SAVED_OBJECT_TYPE, ruleId);
    return {
      error: (0, _lib.createBackfillError)(notFoundError.output.payload.message, ruleId)
    };
  }

  // if rule exists, check that it is enabled
  if (!rule.enabled) {
    return {
      error: (0, _lib.createBackfillError)(`Rule ${ruleId} is disabled`, ruleId, rule.name)
    };
  }

  // check that the rule type is supported
  const isLifecycleRule = (_ruleTypeRegistry$get = ruleTypeRegistry.get(rule.alertTypeId).autoRecoverAlerts) !== null && _ruleTypeRegistry$get !== void 0 ? _ruleTypeRegistry$get : true;
  if (isLifecycleRule) {
    return {
      error: (0, _lib.createBackfillError)(`Rule type "${rule.alertTypeId}" for rule ${ruleId} is not supported`, ruleId, rule.name)
    };
  }

  // check that the API key is not null
  if (!rule.apiKey) {
    return {
      error: (0, _lib.createBackfillError)(`Rule ${ruleId} has no API key`, ruleId, rule.name)
    };
  }
  return {
    rule
  };
}
async function extractRuleActions({
  actionsClient,
  rule,
  runActions
}) {
  var _rule$systemActions;
  // defauts to true if not specified
  const shouldRunActions = runActions !== undefined ? runActions : true;
  if (!shouldRunActions) {
    return {
      hasUnsupportedActions: false,
      actions: [],
      references: []
    };
  }
  const ruleLevelNotifyWhen = rule.notifyWhen;
  const normalizedActions = [];
  for (const action of rule.actions) {
    // if action level frequency is not defined and rule level notifyWhen is, set the action level frequency
    if (!action.frequency && ruleLevelNotifyWhen) {
      normalizedActions.push({
        ...action,
        frequency: {
          notifyWhen: ruleLevelNotifyWhen,
          summary: false,
          throttle: null
        }
      });
    } else {
      normalizedActions.push(action);
    }
  }
  const hasUnsupportedActions = normalizedActions.some(action => {
    var _action$frequency;
    return ((_action$frequency = action.frequency) === null || _action$frequency === void 0 ? void 0 : _action$frequency.notifyWhen) !== 'onActiveAlert';
  });
  const allActions = [...normalizedActions.filter(action => {
    var _action$frequency2;
    return ((_action$frequency2 = action.frequency) === null || _action$frequency2 === void 0 ? void 0 : _action$frequency2.notifyWhen) === 'onActiveAlert';
  }), ...((_rule$systemActions = rule.systemActions) !== null && _rule$systemActions !== void 0 ? _rule$systemActions : [])];
  const {
    references,
    actions
  } = await (0, _denormalize_actions.denormalizeActions)(actionsClient, allActions);
  return {
    hasUnsupportedActions,
    actions,
    references
  };
}