"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useQuerySubscriber = exports.hasQuerySubscriberData = void 0;
var _react = require("react");
var _get_resolved_date_range = require("../utils/get_resolved_date_range");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Memorizes current query, filters and absolute date range
 * @param data
 * @param listenToSearchSessionUpdates
 * @public
 */
const useQuerySubscriber = ({
  data,
  listenToSearchSessionUpdates = true
}) => {
  const timefilter = data.query.timefilter.timefilter;
  const [result, setResult] = (0, _react.useState)(() => {
    const state = data.query.getState();
    const dateRange = (0, _get_resolved_date_range.getResolvedDateRange)(timefilter);
    return {
      query: state === null || state === void 0 ? void 0 : state.query,
      filters: state === null || state === void 0 ? void 0 : state.filters,
      fromDate: dateRange.fromDate,
      toDate: dateRange.toDate
    };
  });
  (0, _react.useEffect)(() => {
    if (!listenToSearchSessionUpdates) {
      return;
    }
    const subscription = data.search.session.state$.subscribe(sessionState => {
      const dateRange = (0, _get_resolved_date_range.getResolvedDateRange)(timefilter);
      setResult(prevState => ({
        ...prevState,
        fromDate: dateRange.fromDate,
        toDate: dateRange.toDate
      }));
    });
    return () => subscription.unsubscribe();
  }, [setResult, timefilter, data.search.session.state$, listenToSearchSessionUpdates]);
  (0, _react.useEffect)(() => {
    if (listenToSearchSessionUpdates) {
      return;
    }
    const subscription = timefilter.getTimeUpdate$().subscribe(() => {
      const dateRange = (0, _get_resolved_date_range.getResolvedDateRange)(timefilter);
      setResult(prevState => ({
        ...prevState,
        fromDate: dateRange.fromDate,
        toDate: dateRange.toDate
      }));
    });
    return () => subscription.unsubscribe();
  }, [setResult, timefilter, listenToSearchSessionUpdates]);
  (0, _react.useEffect)(() => {
    const subscription = data.query.state$.subscribe(({
      state,
      changes
    }) => {
      if (changes.query || changes.filters) {
        setResult(prevState => ({
          ...prevState,
          query: state.query,
          filters: state.filters
        }));
      }
    });
    return () => subscription.unsubscribe();
  }, [setResult, data.query.state$]);
  return result;
};

/**
 * Checks if query result is ready to be used
 * @param result
 * @public
 */
exports.useQuerySubscriber = useQuerySubscriber;
const hasQuerySubscriberData = result => Boolean(result.query && result.filters && result.fromDate && result.toDate);
exports.hasQuerySubscriberData = hasQuerySubscriberData;