"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TelemetryPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _analyticsShippersElasticV3Browser = require("@kbn/analytics-shippers-elastic-v3-browser");
var _rxjs = require("rxjs");
var _routes = require("../common/routes");
var _services = require("./services");
var _render_welcome_telemetry_notice = require("./render_welcome_telemetry_notice");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function getTelemetryConstants(docLinks) {
  return {
    getPrivacyStatementUrl: () => docLinks.links.legal.privacyStatement
  };
}
class TelemetryPlugin {
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "currentKibanaVersion", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "telemetrySender", void 0);
    (0, _defineProperty2.default)(this, "telemetryNotifications", void 0);
    (0, _defineProperty2.default)(this, "telemetryService", void 0);
    (0, _defineProperty2.default)(this, "canUserChangeSettings", true);
    this.currentKibanaVersion = initializerContext.env.packageInfo.version;
    this.config = initializerContext.config.get();
  }
  setup({
    analytics,
    http,
    notifications,
    getStartServices
  }, {
    screenshotMode,
    home
  }) {
    const config = this.config;
    const currentKibanaVersion = this.currentKibanaVersion;
    this.telemetryService = new _services.TelemetryService({
      config,
      isScreenshotMode: screenshotMode.isScreenshotMode(),
      http,
      notifications,
      currentKibanaVersion
    });
    let telemetryConstants;
    getStartServices().then(([{
      docLinks
    }]) => {
      telemetryConstants = getTelemetryConstants(docLinks);
    });
    analytics.registerContextProvider({
      name: 'telemetry labels',
      context$: (0, _rxjs.of)({
        labels: this.config.labels
      }),
      schema: {
        labels: {
          type: 'pass_through',
          _meta: {
            description: 'Custom labels added to the telemetry.labels config in the kibana.yml'
          }
        }
      }
    });
    analytics.registerShipper(_analyticsShippersElasticV3Browser.ElasticV3BrowserShipper, {
      channelName: 'kibana-browser',
      version: currentKibanaVersion,
      sendTo: config.sendUsageTo === 'prod' ? 'production' : 'staging'
    });
    this.telemetrySender = new _services.TelemetrySender(this.telemetryService, async () => {
      await this.refreshConfig(http);
      analytics.optIn({
        global: {
          enabled: this.telemetryService.isOptedIn
        }
      });
    });
    if (home && !this.config.hidePrivacyStatement) {
      home.welcomeScreen.registerOnRendered(() => {
        var _this$telemetryServic;
        if ((_this$telemetryServic = this.telemetryService) !== null && _this$telemetryServic !== void 0 && _this$telemetryServic.userCanChangeSettings) {
          var _this$telemetryNotifi;
          (_this$telemetryNotifi = this.telemetryNotifications) === null || _this$telemetryNotifi === void 0 ? void 0 : _this$telemetryNotifi.setOptInStatusNoticeSeen();
        }
      });
      home.welcomeScreen.registerTelemetryNoticeRenderer(() => (0, _render_welcome_telemetry_notice.renderWelcomeTelemetryNotice)(this.telemetryService, http.basePath.prepend, telemetryConstants));
    }
    return {
      telemetryService: this.getTelemetryServicePublicApis()
    };
  }
  start({
    analytics,
    http,
    overlays,
    theme,
    application,
    docLinks
  }) {
    if (!this.telemetryService) {
      throw Error('Telemetry plugin failed to initialize properly.');
    }
    this.canUserChangeSettings = this.getCanUserChangeSettings(application);
    this.telemetryService.userCanChangeSettings = this.canUserChangeSettings;
    const telemetryConstants = getTelemetryConstants(docLinks);
    const telemetryNotifications = new _services.TelemetryNotifications({
      http,
      overlays,
      theme,
      telemetryService: this.telemetryService,
      telemetryConstants
    });
    this.telemetryNotifications = telemetryNotifications;
    application.currentAppId$.subscribe(async () => {
      const isUnauthenticated = this.getIsUnauthenticated(http);
      if (isUnauthenticated) {
        return;
      }

      // Refresh and get telemetry config
      const updatedConfig = await this.refreshConfig(http);
      analytics.optIn({
        global: {
          enabled: this.telemetryService.isOptedIn
        }
      });
      const telemetryBanner = updatedConfig === null || updatedConfig === void 0 ? void 0 : updatedConfig.banner;
      this.maybeStartTelemetryPoller();
      if (telemetryBanner) {
        this.maybeShowOptedInNotificationBanner();
      }
    });
    return {
      telemetryService: this.getTelemetryServicePublicApis(),
      telemetryNotifications: {
        setOptedInNoticeSeen: () => telemetryNotifications.setOptInStatusNoticeSeen()
      },
      telemetryConstants
    };
  }
  stop() {
    var _this$telemetrySender;
    (_this$telemetrySender = this.telemetrySender) === null || _this$telemetrySender === void 0 ? void 0 : _this$telemetrySender.stop();
  }
  getTelemetryServicePublicApis() {
    const telemetryService = this.telemetryService;
    return {
      getIsOptedIn: () => telemetryService.getIsOptedIn(),
      setOptIn: optedIn => telemetryService.setOptIn(optedIn),
      canSendTelemetry: () => telemetryService.canSendTelemetry(),
      userCanChangeSettings: telemetryService.userCanChangeSettings,
      getCanChangeOptInStatus: () => telemetryService.getCanChangeOptInStatus(),
      fetchExample: () => telemetryService.fetchExample()
    };
  }

  /**
   * Retrieve the up-to-date configuration
   * @param http HTTP helper to make requests to the server
   * @private
   */
  async refreshConfig(http) {
    const updatedConfig = await this.fetchUpdatedConfig(http);
    if (this.telemetryService) {
      this.telemetryService.config = updatedConfig;
    }
    return updatedConfig;
  }

  /**
   * Can the user edit the saved objects?
   * This is a security feature, not included in the OSS build, so we need to fallback to `true`
   * in case it is `undefined`.
   * @param application CoreStart.application
   * @private
   */
  getCanUserChangeSettings(application) {
    var _ref, _application$capabili, _application$capabili2;
    return (_ref = (_application$capabili = application.capabilities) === null || _application$capabili === void 0 ? void 0 : (_application$capabili2 = _application$capabili.savedObjectsManagement) === null || _application$capabili2 === void 0 ? void 0 : _application$capabili2.edit) !== null && _ref !== void 0 ? _ref : true;
  }
  getIsUnauthenticated(http) {
    const {
      anonymousPaths
    } = http;
    return anonymousPaths.isAnonymous(window.location.pathname);
  }
  maybeStartTelemetryPoller() {
    if (!this.telemetrySender) {
      return;
    }
    this.telemetrySender.startChecking();
  }
  maybeShowOptedInNotificationBanner() {
    if (!this.telemetryNotifications) {
      return;
    }
    const shouldShowBanner = this.telemetryNotifications.shouldShowOptInStatusNoticeBanner();
    if (shouldShowBanner) {
      this.telemetryNotifications.renderOptInStatusNoticeBanner();
    }
  }

  /**
   * Fetch configuration from the server and merge it with the one the browser already knows
   * @param http The HTTP helper to make the requests
   * @private
   */
  async fetchUpdatedConfig(http) {
    const {
      allowChangingOptInStatus,
      optIn,
      sendUsageFrom,
      telemetryNotifyUserAboutOptInDefault
    } = await http.get(_routes.FetchTelemetryConfigRoute);
    return {
      ...this.config,
      allowChangingOptInStatus,
      optIn,
      sendUsageFrom,
      telemetryNotifyUserAboutOptInDefault,
      userCanChangeSettings: this.canUserChangeSettings
    };
  }
}
exports.TelemetryPlugin = TelemetryPlugin;