"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SyntheticsPrivateLocation = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _get_agent_policies = require("../../routes/settings/private_locations/get_agent_policies");
var _format_synthetics_policy = require("../formatters/private_formatters/format_synthetics_policy");
var _runtime_types = require("../../../common/runtime_types");
var _formatting_utils = require("../formatters/private_formatters/formatting_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SyntheticsPrivateLocation {
  constructor(_server) {
    (0, _defineProperty2.default)(this, "server", void 0);
    this.server = _server;
  }
  async buildNewPolicy() {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const newPolicy = await this.server.fleet.packagePolicyService.buildPackagePolicyFromPackage(soClient, 'synthetics', this.server.logger);
    if (!newPolicy) {
      throw new Error(`Unable to create Synthetics package policy template for private location`);
    }
    return newPolicy;
  }
  getPolicyId(config, locId, spaceId) {
    if (config[_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE] === _runtime_types.SourceType.PROJECT) {
      return `${config.id}-${locId}`;
    }
    return `${config.id}-${locId}-${spaceId}`;
  }
  generateNewPolicy(config, privateLocation, newPolicyTemplate, spaceId, globalParams) {
    const {
      label: locName
    } = privateLocation;
    const newPolicy = (0, _lodash.cloneDeep)(newPolicyTemplate);
    try {
      var _config$fields, _config$fields2, _config$fields3;
      newPolicy.is_managed = true;
      newPolicy.policy_id = privateLocation.agentPolicyId;
      if (config[_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE] === _runtime_types.SourceType.PROJECT) {
        newPolicy.name = `${config.id}-${locName}`;
      } else {
        newPolicy.name = `${config[_runtime_types.ConfigKey.NAME]}-${locName}-${spaceId}`;
      }
      newPolicy.namespace = config[_runtime_types.ConfigKey.NAMESPACE];
      const {
        formattedPolicy
      } = (0, _format_synthetics_policy.formatSyntheticsPolicy)(newPolicy, config.type, {
        ...config,
        config_id: (_config$fields = config.fields) === null || _config$fields === void 0 ? void 0 : _config$fields.config_id,
        location_name: (0, _formatting_utils.stringifyString)(privateLocation.label),
        location_id: privateLocation.id,
        'monitor.project.id': (0, _formatting_utils.stringifyString)((_config$fields2 = config.fields) === null || _config$fields2 === void 0 ? void 0 : _config$fields2['monitor.project.name']),
        'monitor.project.name': (0, _formatting_utils.stringifyString)((_config$fields3 = config.fields) === null || _config$fields3 === void 0 ? void 0 : _config$fields3['monitor.project.name'])
      }, globalParams);
      return formattedPolicy;
    } catch (e) {
      this.server.logger.error(e);
      return null;
    }
  }
  async createPackagePolicies(configs, request, privateLocations, spaceId) {
    if (configs.length === 0) {
      return {
        created: [],
        failed: []
      };
    }
    const newPolicies = [];
    const newPolicyTemplate = await this.buildNewPolicy();
    for (const {
      config,
      globalParams
    } of configs) {
      try {
        const {
          locations
        } = config;
        const fleetManagedLocations = locations.filter(loc => !loc.isServiceManaged);
        for (const privateLocation of fleetManagedLocations) {
          const location = privateLocations === null || privateLocations === void 0 ? void 0 : privateLocations.find(loc => loc.id === privateLocation.id);
          if (!location) {
            throw new Error(`Unable to find Synthetics private location for agentId ${privateLocation.id}`);
          }
          const newPolicy = this.generateNewPolicy(config, location, newPolicyTemplate, spaceId, globalParams);
          if (!newPolicy) {
            throw new Error(`Unable to create Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]} with private location ${location.label}`);
          }
          if (newPolicy) {
            newPolicies.push({
              ...newPolicy,
              id: this.getPolicyId(config, location.id, spaceId)
            });
          }
        }
      } catch (e) {
        this.server.logger.error(e);
        throw e;
      }
    }
    if (newPolicies.length === 0) {
      throw new Error('Failed to build package policies for all monitors');
    }
    try {
      return await this.createPolicyBulk(newPolicies);
    } catch (e) {
      this.server.logger.error(e);
      throw e;
    }
  }
  async inspectPackagePolicy({
    privateConfig,
    spaceId,
    allPrivateLocations
  }) {
    if (!privateConfig) {
      return null;
    }
    const newPolicyTemplate = await this.buildNewPolicy();
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const {
      config,
      globalParams
    } = privateConfig;
    try {
      const {
        locations
      } = config;
      const privateLocation = locations.find(loc => !loc.isServiceManaged);
      const location = allPrivateLocations === null || allPrivateLocations === void 0 ? void 0 : allPrivateLocations.find(loc => loc.id === (privateLocation === null || privateLocation === void 0 ? void 0 : privateLocation.id));
      const newPolicy = this.generateNewPolicy(config, location, newPolicyTemplate, spaceId, globalParams);
      const pkgPolicy = {
        ...newPolicy,
        id: this.getPolicyId(config, location.id, spaceId)
      };
      return await this.server.fleet.packagePolicyService.inspect(soClient, pkgPolicy);
    } catch (e) {
      this.server.logger.error(e);
      return null;
    }
  }
  async editMonitors(configs, request, allPrivateLocations, spaceId) {
    if (configs.length === 0) {
      return {};
    }
    const newPolicyTemplate = await this.buildNewPolicy();
    const policiesToUpdate = [];
    const policiesToCreate = [];
    const policiesToDelete = [];
    const existingPolicies = await this.getExistingPolicies(configs.map(({
      config
    }) => config), allPrivateLocations, spaceId);
    for (const {
      config,
      globalParams
    } of configs) {
      const {
        locations
      } = config;
      const monitorPrivateLocations = locations.filter(loc => !loc.isServiceManaged);
      for (const privateLocation of allPrivateLocations) {
        const hasLocation = monitorPrivateLocations === null || monitorPrivateLocations === void 0 ? void 0 : monitorPrivateLocations.some(loc => loc.id === privateLocation.id);
        const currId = this.getPolicyId(config, privateLocation.id, spaceId);
        const hasPolicy = existingPolicies === null || existingPolicies === void 0 ? void 0 : existingPolicies.some(policy => policy.id === currId);
        try {
          if (hasLocation) {
            const newPolicy = this.generateNewPolicy(config, privateLocation, newPolicyTemplate, spaceId, globalParams);
            if (!newPolicy) {
              throwAddEditError(hasPolicy, privateLocation.label);
            }
            if (hasPolicy) {
              policiesToUpdate.push({
                ...newPolicy,
                id: currId
              });
            } else {
              policiesToCreate.push({
                ...newPolicy,
                id: currId
              });
            }
          } else if (hasPolicy) {
            policiesToDelete.push(currId);
          }
        } catch (e) {
          this.server.logger.error(e);
          throwAddEditError(hasPolicy, privateLocation.label, config[_runtime_types.ConfigKey.NAME]);
        }
      }
    }
    const [_createResponse, failedUpdatesRes, _deleteResponse] = await Promise.all([this.createPolicyBulk(policiesToCreate), this.updatePolicyBulk(policiesToUpdate), this.deletePolicyBulk(policiesToDelete)]);
    const failedUpdates = failedUpdatesRes === null || failedUpdatesRes === void 0 ? void 0 : failedUpdatesRes.map(({
      packagePolicy,
      error
    }) => {
      const policyConfig = configs.find(({
        config
      }) => {
        const {
          locations
        } = config;
        const monitorPrivateLocations = locations.filter(loc => !loc.isServiceManaged);
        for (const privateLocation of monitorPrivateLocations) {
          const currId = this.getPolicyId(config, privateLocation.id, spaceId);
          return currId === packagePolicy.id;
        }
      });
      return {
        error,
        packagePolicy,
        config: policyConfig === null || policyConfig === void 0 ? void 0 : policyConfig.config
      };
    });
    return {
      failedUpdates
    };
  }
  async getExistingPolicies(configs, allPrivateLocations, spaceId) {
    var _await$this$server$fl;
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const listOfPolicies = [];
    for (const config of configs) {
      for (const privateLocation of allPrivateLocations) {
        const currId = this.getPolicyId(config, privateLocation.id, spaceId);
        listOfPolicies.push(currId);
      }
    }
    return (_await$this$server$fl = await this.server.fleet.packagePolicyService.getByIDs(soClient, listOfPolicies, {
      ignoreMissing: true
    })) !== null && _await$this$server$fl !== void 0 ? _await$this$server$fl : [];
  }
  async createPolicyBulk(newPolicies) {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const esClient = this.server.uptimeEsClient.baseESClient;
    if (esClient && newPolicies.length > 0) {
      return await this.server.fleet.packagePolicyService.bulkCreate(soClient, esClient, newPolicies);
    }
  }
  async updatePolicyBulk(policiesToUpdate) {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const esClient = this.server.uptimeEsClient.baseESClient;
    if (soClient && esClient && policiesToUpdate.length > 0) {
      const {
        failedPolicies
      } = await this.server.fleet.packagePolicyService.bulkUpdate(soClient, esClient, policiesToUpdate, {
        force: true
      });
      return failedPolicies;
    }
  }
  async deletePolicyBulk(policyIdsToDelete) {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const esClient = this.server.uptimeEsClient.baseESClient;
    if (soClient && esClient && policyIdsToDelete.length > 0) {
      return await this.server.fleet.packagePolicyService.delete(soClient, esClient, policyIdsToDelete, {
        force: true
      });
    }
  }
  async deleteMonitors(configs, request, spaceId) {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const esClient = this.server.uptimeEsClient.baseESClient;
    const policyIdsToDelete = [];
    for (const config of configs) {
      const {
        locations
      } = config;
      const monitorPrivateLocations = locations.filter(loc => !loc.isServiceManaged);
      for (const privateLocation of monitorPrivateLocations) {
        try {
          policyIdsToDelete.push(this.getPolicyId(config, privateLocation.id, spaceId));
        } catch (e) {
          this.server.logger.error(e);
          throw new Error(deletePolicyError(config[_runtime_types.ConfigKey.NAME], privateLocation.label));
        }
      }
    }
    if (policyIdsToDelete.length > 0) {
      const result = await this.server.fleet.packagePolicyService.delete(soClient, esClient, policyIdsToDelete, {
        force: true
      });
      const failedPolicies = result === null || result === void 0 ? void 0 : result.filter(policy => !policy.success);
      if ((failedPolicies === null || failedPolicies === void 0 ? void 0 : failedPolicies.length) === policyIdsToDelete.length) {
        throw new Error(deletePolicyError(configs[0][_runtime_types.ConfigKey.NAME]));
      }
      return result;
    }
  }
  async getAgentPolicies() {
    const agentPolicies = await (0, _get_agent_policies.getAgentPoliciesAsInternalUser)(this.server);
    return agentPolicies.items;
  }
}
exports.SyntheticsPrivateLocation = SyntheticsPrivateLocation;
const throwAddEditError = (hasPolicy, location, name) => {
  throw new Error(`Unable to ${hasPolicy ? 'update' : 'create'} Synthetics package policy ${name ? 'for monitor ' + name : ''} for private location: ${location}`);
};
const deletePolicyError = (name, location) => {
  return `Unable to delete Synthetics package policy for monitor ${name} with private location ${location}`;
};