"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncNewMonitorBulk = exports.createNewSavedObjectMonitorBulk = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _uuid = require("uuid");
var _add_monitor = require("../add_monitor");
var _monitor_upgrade_sender = require("../../telemetry/monitor_upgrade_sender");
var _delete_monitor = require("../delete_monitor");
var _utils = require("../../../synthetics_service/utils");
var _saved_objects = require("../../../../common/types/saved_objects");
var _runtime_types = require("../../../../common/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createNewSavedObjectMonitorBulk = async ({
  soClient,
  monitorsToCreate
}) => {
  const newMonitors = monitorsToCreate.map(({
    id,
    monitor
  }) => ({
    id,
    type: _saved_objects.syntheticsMonitorType,
    attributes: (0, _utils.formatSecrets)({
      ...monitor,
      [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: monitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || id,
      [_runtime_types.ConfigKey.CONFIG_ID]: id,
      revision: 1
    })
  }));
  const result = await soClient.bulkCreate(newMonitors);
  return result.saved_objects;
};
exports.createNewSavedObjectMonitorBulk = createNewSavedObjectMonitorBulk;
const syncNewMonitorBulk = async ({
  routeContext,
  normalizedMonitors,
  privateLocations,
  spaceId
}) => {
  const {
    server,
    savedObjectsClient,
    syntheticsMonitorClient,
    request
  } = routeContext;
  let newMonitors = null;
  const monitorsToCreate = normalizedMonitors.map(monitor => {
    const monitorSavedObjectId = (0, _uuid.v4)();
    return {
      id: monitorSavedObjectId,
      monitor: {
        ...monitor,
        [_runtime_types.ConfigKey.CONFIG_ID]: monitorSavedObjectId,
        [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: monitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || monitorSavedObjectId
      }
    };
  });
  try {
    const [createdMonitors, [policiesResult, syncErrors]] = await Promise.all([createNewSavedObjectMonitorBulk({
      monitorsToCreate,
      soClient: savedObjectsClient
    }), syntheticsMonitorClient.addMonitors(monitorsToCreate, request, savedObjectsClient, privateLocations, spaceId)]);
    let failedMonitors = [];
    const {
      failed: failedPolicies
    } = policiesResult !== null && policiesResult !== void 0 ? policiesResult : {};
    newMonitors = createdMonitors;
    if (failedPolicies && (failedPolicies === null || failedPolicies === void 0 ? void 0 : failedPolicies.length) > 0 && newMonitors) {
      failedMonitors = await handlePrivateConfigErrors(routeContext, newMonitors, failedPolicies);
    }
    sendNewMonitorTelemetry(server, newMonitors, syncErrors);
    return {
      errors: syncErrors,
      newMonitors,
      failedMonitors
    };
  } catch (e) {
    await rollBackNewMonitorBulk(monitorsToCreate, routeContext);
    throw e;
  }
};
exports.syncNewMonitorBulk = syncNewMonitorBulk;
const handlePrivateConfigErrors = async (routeContext, createdMonitors, failedPolicies) => {
  const failedMonitors = [];
  await (0, _pMap.default)(failedPolicies, async ({
    packagePolicy,
    error
  }) => {
    var _enabledInput$streams, _vars$ConfigKey$CONFI;
    const {
      inputs
    } = packagePolicy;
    const enabledInput = inputs === null || inputs === void 0 ? void 0 : inputs.find(input => input.enabled);
    const stream = enabledInput === null || enabledInput === void 0 ? void 0 : (_enabledInput$streams = enabledInput.streams) === null || _enabledInput$streams === void 0 ? void 0 : _enabledInput$streams[0];
    const vars = stream === null || stream === void 0 ? void 0 : stream.vars;
    const monitorId = vars === null || vars === void 0 ? void 0 : (_vars$ConfigKey$CONFI = vars[_runtime_types.ConfigKey.CONFIG_ID]) === null || _vars$ConfigKey$CONFI === void 0 ? void 0 : _vars$ConfigKey$CONFI.value;
    const monitor = createdMonitors.find(savedObject => savedObject.attributes[_runtime_types.ConfigKey.CONFIG_ID] === monitorId);
    if (monitor) {
      failedMonitors.push({
        monitor,
        error
      });
      await (0, _add_monitor.deleteMonitorIfCreated)({
        routeContext,
        newMonitorId: monitor.id
      });
      createdMonitors.splice(createdMonitors.indexOf(monitor), 1);
    }
  });
  return failedMonitors;
};
const rollBackNewMonitorBulk = async (monitorsToCreate, routeContext) => {
  const {
    server
  } = routeContext;
  try {
    await (0, _pMap.default)(monitorsToCreate, async monitor => (0, _delete_monitor.deleteMonitor)({
      routeContext,
      monitorId: monitor.id
    }), {
      concurrency: 100,
      stopOnError: false
    });
  } catch (e) {
    // ignore errors here
    server.logger.error(e);
  }
};
const sendNewMonitorTelemetry = (server, monitors, errors) => {
  for (const monitor of monitors) {
    (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryEvent)({
      errors,
      monitor,
      isInlineScript: Boolean(monitor.attributes[_runtime_types.ConfigKey.SOURCE_INLINE]),
      stackVersion: server.stackVersion
    }));
  }
};