"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteAnomalyAlertAction = exports.deleteAlertAction = exports.createAlertAction = exports.connectorsSelector = exports.anomalyAlertSelector = exports.alertsSelector = exports.alertsReducer = void 0;
exports.fetchAlertsEffect = fetchAlertsEffect;
exports.newAlertSelector = exports.isAnomalyAlertDeleting = exports.isAlertDeletedSelector = exports.getMonitorAlertsAction = exports.getConnectorsAction = exports.getAnomalyAlertAction = void 0;
var _i18n = require("@kbn/i18n");
var _reduxActions = require("redux-actions");
var _effects = require("redux-saga/effects");
var _utils = require("../actions/utils");
var _utils2 = require("../reducers/utils");
var _fetch_effect = require("../effects/fetch_effect");
var _alerts = require("../api/alerts");
var _kibana_service = require("../kibana_service");
var _selectors = require("../selectors");
var _alert_messages = require("../../lib/alert_types/alert_messages");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createAlertAction = (0, _utils.createAsyncAction)('CREATE ALERT');
exports.createAlertAction = createAlertAction;
const getConnectorsAction = (0, _utils.createAsyncAction)('GET CONNECTORS');
exports.getConnectorsAction = getConnectorsAction;
const getMonitorAlertsAction = (0, _utils.createAsyncAction)('GET ALERTS');
exports.getMonitorAlertsAction = getMonitorAlertsAction;
const getAnomalyAlertAction = (0, _utils.createAsyncAction)('GET EXISTING ALERTS');
exports.getAnomalyAlertAction = getAnomalyAlertAction;
const deleteAlertAction = (0, _utils.createAsyncAction)('DELETE ALERTS');
exports.deleteAlertAction = deleteAlertAction;
const deleteAnomalyAlertAction = (0, _utils.createAsyncAction)('DELETE ANOMALY ALERT');
exports.deleteAnomalyAlertAction = deleteAnomalyAlertAction;
const initialState = {
  connectors: (0, _utils2.asyncInitState)(),
  newAlert: (0, _utils2.asyncInitState)(),
  alerts: (0, _utils2.asyncInitState)(),
  anomalyAlert: (0, _utils2.asyncInitState)(),
  alertDeletion: (0, _utils2.asyncInitState)(),
  anomalyAlertDeletion: (0, _utils2.asyncInitState)()
};
const alertsReducer = (0, _reduxActions.handleActions)({
  ...(0, _utils2.handleAsyncAction)('connectors', getConnectorsAction),
  ...(0, _utils2.handleAsyncAction)('newAlert', createAlertAction),
  ...(0, _utils2.handleAsyncAction)('alerts', getMonitorAlertsAction),
  ...(0, _utils2.handleAsyncAction)('anomalyAlert', getAnomalyAlertAction),
  ...(0, _utils2.handleAsyncAction)('alertDeletion', deleteAlertAction),
  ...(0, _utils2.handleAsyncAction)('anomalyAlertDeletion', deleteAnomalyAlertAction)
}, initialState);
exports.alertsReducer = alertsReducer;
const showAlertDisabledSuccess = () => {
  _kibana_service.kibanaService.core.notifications.toasts.addSuccess(_i18n.i18n.translate('xpack.synthetics.overview.alerts.disabled.success', {
    defaultMessage: 'Rule successfully disabled!'
  }));
};
const showAlertDisabledFailed = err => {
  _kibana_service.kibanaService.core.notifications.toasts.addError(err, {
    title: _i18n.i18n.translate('xpack.synthetics.overview.alerts.disabled.failed', {
      defaultMessage: 'Rule cannot be disabled!'
    })
  });
};
function* fetchAlertsEffect() {
  yield (0, _effects.takeLatest)(getAnomalyAlertAction.get, (0, _fetch_effect.fetchEffectFactory)(_alerts.fetchAnomalyAlertRecords, getAnomalyAlertAction.success, getAnomalyAlertAction.fail));
  yield (0, _effects.takeLatest)(deleteAnomalyAlertAction.get, function* (action) {
    try {
      yield (0, _effects.call)(_alerts.disableAlertById, action.payload);
      yield (0, _effects.put)(deleteAnomalyAlertAction.success(action.payload.alertId));
      showAlertDisabledSuccess();
      const monitorId = yield (0, _effects.select)(_selectors.monitorIdSelector);
      yield (0, _effects.put)(getAnomalyAlertAction.get({
        monitorId
      }));
    } catch (err) {
      showAlertDisabledFailed(err);
      yield (0, _effects.put)(deleteAnomalyAlertAction.fail(err));
    }
  });
  yield (0, _effects.takeLatest)(deleteAlertAction.get, function* (action) {
    try {
      yield (0, _effects.call)(_alerts.disableAlertById, action.payload);
      // clear previous state
      yield (0, _effects.put)(createAlertAction.success(null));
      yield (0, _effects.put)(deleteAlertAction.success(action.payload.alertId));
      showAlertDisabledSuccess();
      yield (0, _effects.put)(getMonitorAlertsAction.get());
    } catch (err) {
      showAlertDisabledFailed(err);
      yield (0, _effects.put)(deleteAlertAction.fail(err));
    }
  });
  yield (0, _effects.takeLatest)(getConnectorsAction.get, (0, _fetch_effect.fetchEffectFactory)(_alerts.fetchConnectors, getConnectorsAction.success, getConnectorsAction.fail));
  yield (0, _effects.takeLatest)(getMonitorAlertsAction.get, (0, _fetch_effect.fetchEffectFactory)(_alerts.fetchMonitorAlertRecords, getMonitorAlertsAction.success, getMonitorAlertsAction.fail));
  yield (0, _effects.takeLatest)(createAlertAction.get, function* (action) {
    try {
      const response = yield (0, _effects.call)(_alerts.createAlert, action.payload);
      yield (0, _effects.put)(createAlertAction.success(response));
      _kibana_service.kibanaService.core.notifications.toasts.addSuccess((0, _alert_messages.simpleAlertEnabled)(action.payload.defaultActions, _kibana_service.kibanaService.theme, response));
      yield (0, _effects.put)(getMonitorAlertsAction.get());
    } catch (err) {
      _kibana_service.kibanaService.core.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('xpack.synthetics.overview.alerts.enabled.failed', {
          defaultMessage: 'Rule cannot be enabled!'
        })
      });
      yield (0, _effects.put)(createAlertAction.fail(err));
    }
  });
}
const connectorsSelector = ({
  alerts
}) => alerts.connectors;
exports.connectorsSelector = connectorsSelector;
const newAlertSelector = ({
  alerts
}) => alerts.newAlert;
exports.newAlertSelector = newAlertSelector;
const alertsSelector = ({
  alerts
}) => alerts.alerts;
exports.alertsSelector = alertsSelector;
const isAlertDeletedSelector = ({
  alerts
}) => alerts.alertDeletion;
exports.isAlertDeletedSelector = isAlertDeletedSelector;
const anomalyAlertSelector = ({
  alerts
}) => alerts.anomalyAlert;
exports.anomalyAlertSelector = anomalyAlertSelector;
const isAnomalyAlertDeleting = ({
  alerts
}) => alerts.anomalyAlertDeletion.loading;
exports.isAnomalyAlertDeleting = isAnomalyAlertDeleting;