"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createExternalService = void 0;
var _axios = _interopRequireDefault(require("axios"));
var _i18n = require("@kbn/i18n");
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _pipeable = require("fp-ts/lib/pipeable");
var _Option = require("fp-ts/lib/Option");
var _translations = require("./translations");
var _lib = require("../../../common/slack_api/lib");
var _constants = require("../../../common/slack_api/constants");
var _http_response_retry_header = require("../lib/http_response_retry_header");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RE_TRY = 5;
const LIMIT = 1000;
const buildSlackExecutorErrorResponse = ({
  slackApiError,
  logger
}) => {
  if (!slackApiError.response) {
    return (0, _lib.serviceErrorResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiError.message);
  }
  const {
    status,
    statusText,
    headers
  } = slackApiError.response;

  // special handling for 5xx
  if (status >= 500) {
    return (0, _lib.retryResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiError.message);
  }

  // special handling for rate limiting
  if (status === 429) {
    return (0, _pipeable.pipe)((0, _http_response_retry_header.getRetryAfterIntervalFromHeaders)(headers), (0, _Option.map)(retry => (0, _lib.retryResultSeconds)(_constants.SLACK_API_CONNECTOR_ID, slackApiError.message, retry)), (0, _Option.getOrElse)(() => (0, _lib.retryResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiError.message)));
  }
  const errorMessage = _i18n.i18n.translate('xpack.stackConnectors.slack.unexpectedHttpResponseErrorMessage', {
    defaultMessage: 'unexpected http response from slack: {httpStatus} {httpStatusText}',
    values: {
      httpStatus: status,
      httpStatusText: statusText
    }
  });
  logger.error(`error on ${_constants.SLACK_API_CONNECTOR_ID} slack action: ${errorMessage}`);
  return (0, _lib.errorResult)(_constants.SLACK_API_CONNECTOR_ID, errorMessage);
};
const buildSlackExecutorSuccessResponse = ({
  slackApiResponseData
}) => {
  if (!slackApiResponseData) {
    const errMessage = _i18n.i18n.translate('xpack.stackConnectors.slack.unexpectedNullResponseErrorMessage', {
      defaultMessage: 'unexpected null response from slack'
    });
    return (0, _lib.errorResult)(_constants.SLACK_API_CONNECTOR_ID, errMessage);
  }
  if (!slackApiResponseData.ok) {
    return (0, _lib.serviceErrorResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiResponseData.error);
  }
  return (0, _lib.successResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiResponseData);
};
const createExternalService = ({
  secrets
}, logger, configurationUtilities) => {
  const {
    token
  } = secrets;
  if (!token) {
    throw Error(`[Action][${_translations.SLACK_CONNECTOR_NAME}]: Wrong configuration.`);
  }
  const axiosInstance = _axios.default.create({
    baseURL: _constants.SLACK_URL,
    headers: {
      Authorization: `Bearer ${token}`,
      'Content-type': 'application/json; charset=UTF-8'
    }
  });
  const getChannels = async () => {
    try {
      const fetchChannels = (cursor = '') => {
        return (0, _axios_utils.request)({
          axios: axiosInstance,
          configurationUtilities,
          logger,
          method: 'get',
          url: `conversations.list?exclude_archived=true&types=public_channel,private_channel&limit=${LIMIT}${cursor.length > 0 ? `&cursor=${cursor}` : ''}`
        });
      };
      let numberOfFetch = 0;
      let cursor = '';
      const channels = [];
      let result = {
        data: {
          ok: false,
          channels
        },
        status: 0,
        statusText: '',
        headers: {},
        config: {}
      };
      while (numberOfFetch < RE_TRY) {
        var _result$data$channels, _result$data;
        result = await fetchChannels(cursor);
        if (result.data.ok && ((_result$data$channels = (_result$data = result.data) === null || _result$data === void 0 ? void 0 : _result$data.channels) !== null && _result$data$channels !== void 0 ? _result$data$channels : []).length > 0) {
          var _result$data$channels2, _result$data2;
          channels.push(...((_result$data$channels2 = (_result$data2 = result.data) === null || _result$data2 === void 0 ? void 0 : _result$data2.channels) !== null && _result$data$channels2 !== void 0 ? _result$data$channels2 : []));
        }
        if (result.data.ok && result.data.response_metadata && result.data.response_metadata.next_cursor && result.data.response_metadata.next_cursor.length > 0) {
          numberOfFetch += 1;
          cursor = result.data.response_metadata.next_cursor;
        } else {
          break;
        }
      }
      result.data.channels = channels;
      const responseData = result.data;
      return buildSlackExecutorSuccessResponse({
        slackApiResponseData: responseData
      });
    } catch (error) {
      return buildSlackExecutorErrorResponse({
        slackApiError: error,
        logger
      });
    }
  };
  const postMessage = async ({
    channels,
    text
  }) => {
    try {
      const result = await (0, _axios_utils.request)({
        axios: axiosInstance,
        method: 'post',
        url: 'chat.postMessage',
        logger,
        data: {
          channel: channels[0],
          text
        },
        configurationUtilities
      });
      return buildSlackExecutorSuccessResponse({
        slackApiResponseData: result.data
      });
    } catch (error) {
      return buildSlackExecutorErrorResponse({
        slackApiError: error,
        logger
      });
    }
  };
  return {
    getChannels,
    postMessage
  };
};
exports.createExternalService = createExternalService;