"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executor = executor;
exports.getChecksum = getChecksum;
exports.getInvalidComparatorError = getInvalidComparatorError;
exports.getSearchParams = getSearchParams;
exports.getValidTimefieldSort = getValidTimefieldSort;
exports.tryToParseAsDate = tryToParseAsDate;
var _jsSha = require("js-sha256");
var _i18n = require("@kbn/i18n");
var _server = require("@kbn/alerting-plugin/server");
var _common = require("@kbn/triggers-actions-ui-plugin/common");
var _common2 = require("../../../common");
var _action_context = require("./action_context");
var _constants = require("./constants");
var _fetch_es_query = require("./lib/fetch_es_query");
var _fetch_search_source_query = require("./lib/fetch_search_source_query");
var _util = require("./util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function executor(core, options) {
  var _core$http$basePath$p;
  const esQueryRule = (0, _util.isEsQueryRule)(options.params.searchType);
  const {
    rule: {
      id: ruleId,
      name
    },
    services,
    params,
    state,
    spaceId,
    logger
  } = options;
  const {
    alertFactory,
    scopedClusterClient,
    searchSourceClient,
    share,
    dataViews
  } = services;
  const currentTimestamp = new Date().toISOString();
  const publicBaseUrl = (_core$http$basePath$p = core.http.basePath.publicBaseUrl) !== null && _core$http$basePath$p !== void 0 ? _core$http$basePath$p : '';
  const spacePrefix = spaceId !== 'default' ? `/s/${spaceId}` : '';
  const alertLimit = alertFactory.alertLimit.getValue();
  const compareFn = _common2.ComparatorFns.get(params.thresholdComparator);
  if (compareFn == null) {
    throw new Error(getInvalidComparatorError(params.thresholdComparator));
  }
  const isGroupAgg = (0, _common.isGroupAggregation)(params.termField);
  // For ungrouped queries, we run the configured query during each rule run, get a hit count
  // and retrieve up to params.size hits. We evaluate the threshold condition using the
  // value of the hit count. If the threshold condition is met, the hits are counted
  // toward the query match and we update the rule state with the timestamp of the latest hit.
  // In the next run of the rule, the latestTimestamp will be used to gate the query in order to
  // avoid counting a document multiple times.
  // latestTimestamp will be ignored if set for grouped queries
  let latestTimestamp = tryToParseAsDate(state.latestTimestamp);
  const {
    parsedResults,
    dateStart,
    dateEnd,
    link
  } = esQueryRule ? await (0, _fetch_es_query.fetchEsQuery)({
    ruleId,
    name,
    alertLimit,
    params: params,
    timestamp: latestTimestamp,
    publicBaseUrl,
    spacePrefix,
    services: {
      scopedClusterClient,
      logger
    }
  }) : await (0, _fetch_search_source_query.fetchSearchSourceQuery)({
    ruleId,
    alertLimit,
    params: params,
    latestTimestamp,
    spacePrefix,
    services: {
      share,
      searchSourceClient,
      logger,
      dataViews
    }
  });
  const unmetGroupValues = {};
  for (const result of parsedResults.results) {
    var _result$value;
    const alertId = result.group;
    const value = (_result$value = result.value) !== null && _result$value !== void 0 ? _result$value : result.count;

    // group aggregations use the bucket selector agg to compare conditions
    // within the ES query, so only 'met' results are returned, therefore we don't need
    // to use the compareFn
    const met = isGroupAgg ? true : compareFn(value, params.threshold);
    if (!met) {
      unmetGroupValues[alertId] = value;
      continue;
    }
    const baseContext = {
      title: name,
      date: currentTimestamp,
      value,
      hits: result.hits,
      link
    };
    const baseActiveContext = {
      ...baseContext,
      conditions: (0, _action_context.getContextConditionsDescription)({
        comparator: params.thresholdComparator,
        threshold: params.threshold,
        aggType: params.aggType,
        aggField: params.aggField,
        ...(isGroupAgg ? {
          group: alertId
        } : {})
      })
    };
    const actionContext = (0, _action_context.addMessages)({
      ruleName: name,
      baseContext: baseActiveContext,
      params,
      ...(isGroupAgg ? {
        group: alertId
      } : {})
    });
    const alert = alertFactory.create(alertId === _common.UngroupedGroupId && !isGroupAgg ? _constants.ConditionMetAlertInstanceId : alertId);
    alert
    // store the params we would need to recreate the query that led to this alert instance
    .replaceState({
      latestTimestamp,
      dateStart,
      dateEnd
    }).scheduleActions(_constants.ActionGroupId, actionContext);
    if (!isGroupAgg) {
      var _result$hits$find;
      // update the timestamp based on the current search results
      const firstValidTimefieldSort = getValidTimefieldSort((_result$hits$find = result.hits.find(hit => getValidTimefieldSort(hit.sort))) === null || _result$hits$find === void 0 ? void 0 : _result$hits$find.sort);
      if (firstValidTimefieldSort) {
        latestTimestamp = firstValidTimefieldSort;
      }
    }
  }
  alertFactory.alertLimit.setLimitReached(parsedResults.truncated);
  const {
    getRecoveredAlerts
  } = alertFactory.done();
  for (const recoveredAlert of getRecoveredAlerts()) {
    var _unmetGroupValues$ale;
    const alertId = recoveredAlert.getId();
    const baseRecoveryContext = {
      title: name,
      date: currentTimestamp,
      value: (_unmetGroupValues$ale = unmetGroupValues[alertId]) !== null && _unmetGroupValues$ale !== void 0 ? _unmetGroupValues$ale : 0,
      hits: [],
      link,
      conditions: (0, _action_context.getContextConditionsDescription)({
        comparator: params.thresholdComparator,
        threshold: params.threshold,
        isRecovered: true,
        aggType: params.aggType,
        aggField: params.aggField,
        ...(isGroupAgg ? {
          group: alertId
        } : {})
      })
    };
    const recoveryContext = (0, _action_context.addMessages)({
      ruleName: name,
      baseContext: baseRecoveryContext,
      params,
      isRecovered: true,
      ...(isGroupAgg ? {
        group: alertId
      } : {})
    });
    recoveredAlert.setContext(recoveryContext);
  }
  return {
    state: {
      latestTimestamp
    }
  };
}
function getInvalidWindowSizeError(windowValue) {
  return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidWindowSizeErrorMessage', {
    defaultMessage: 'invalid format for windowSize: "{windowValue}"',
    values: {
      windowValue
    }
  });
}
function getInvalidQueryError(query) {
  return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidQueryErrorMessage', {
    defaultMessage: 'invalid query specified: "{query}" - query must be JSON',
    values: {
      query
    }
  });
}
function getSearchParams(queryParams) {
  const date = Date.now();
  const {
    esQuery,
    timeWindowSize,
    timeWindowUnit
  } = queryParams;
  let parsedQuery;
  try {
    parsedQuery = JSON.parse(esQuery);
  } catch (err) {
    throw new Error(getInvalidQueryError(esQuery));
  }
  if (parsedQuery && !parsedQuery.query) {
    throw new Error(getInvalidQueryError(esQuery));
  }
  const window = `${timeWindowSize}${timeWindowUnit}`;
  let timeWindow;
  try {
    timeWindow = (0, _server.parseDuration)(window);
  } catch (err) {
    throw new Error(getInvalidWindowSizeError(window));
  }
  const dateStart = new Date(date - timeWindow).toISOString();
  const dateEnd = new Date(date).toISOString();
  return {
    parsedQuery,
    dateStart,
    dateEnd
  };
}
function getValidTimefieldSort(sortValues = []) {
  for (const sortValue of sortValues) {
    const sortDate = tryToParseAsDate(sortValue);
    if (sortDate) {
      return sortDate;
    }
  }
}
function tryToParseAsDate(sortValue) {
  const sortDate = typeof sortValue === 'string' ? Date.parse(sortValue) : sortValue;
  if (sortDate && !isNaN(sortDate)) {
    return new Date(sortDate).toISOString();
  }
}
function getChecksum(params) {
  return _jsSha.sha256.create().update(JSON.stringify(params));
}
function getInvalidComparatorError(comparator) {
  return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidComparatorErrorMessage', {
    defaultMessage: 'invalid thresholdComparator specified: {comparator}',
    values: {
      comparator
    }
  });
}