"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateExpression = exports.hasExpressionValidationErrors = void 0;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/triggers-actions-ui-plugin/public");
var _util = require("./util");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validateCommonParams = ruleParams => {
  const {
    size,
    threshold,
    timeWindowSize,
    thresholdComparator,
    aggType,
    aggField,
    groupBy,
    termSize,
    termField
  } = ruleParams;
  const errors = (0, _lodash.defaultsDeep)({}, _constants.COMMON_EXPRESSION_ERRORS);
  if (!('index' in ruleParams) && !ruleParams.searchType) {
    errors.searchType.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredSearchType', {
      defaultMessage: 'Query type is required.'
    }));
    return errors;
  }
  if (aggType && _public.builtInAggregationTypes[aggType].fieldRequired && !aggField) {
    errors.aggField.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredAggFieldText', {
      defaultMessage: 'Aggregation field is required.'
    }));
  }
  if (groupBy && _public.builtInGroupByTypes[groupBy] && _public.builtInGroupByTypes[groupBy].sizeRequired && !termSize) {
    errors.termSize.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredTermSizedText', {
      defaultMessage: 'Term size is required.'
    }));
  }
  if (groupBy && _public.builtInGroupByTypes[groupBy].validNormalizedTypes && _public.builtInGroupByTypes[groupBy].validNormalizedTypes.length > 0 && !termField) {
    errors.termField.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredTermFieldText', {
      defaultMessage: 'Term field is required.'
    }));
  }
  if (!threshold || threshold.length === 0 || threshold[0] === undefined) {
    errors.threshold0.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredThreshold0Text', {
      defaultMessage: 'Threshold 0 is required.'
    }));
  }
  if (thresholdComparator && _public.builtInComparators[thresholdComparator].requiredValues > 1 && (!threshold || threshold[1] === undefined || threshold && threshold.length < _public.builtInComparators[thresholdComparator].requiredValues)) {
    errors.threshold1.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredThreshold1Text', {
      defaultMessage: 'Threshold 1 is required.'
    }));
  }
  if (threshold && threshold.length === 2 && threshold[0] > threshold[1]) {
    errors.threshold1.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.greaterThenThreshold0Text', {
      defaultMessage: 'Threshold 1 must be > Threshold 0.'
    }));
  }
  if (!timeWindowSize) {
    errors.timeWindowSize.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredTimeWindowSizeText', {
      defaultMessage: 'Time window size is required.'
    }));
  }
  if ((0, _lodash.isNil)(size)) {
    errors.size.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredSizeText', {
      defaultMessage: 'Size is required.'
    }));
  }
  if (size && size < 0 || size > 10000) {
    errors.size.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.invalidSizeRangeText', {
      defaultMessage: 'Size must be between 0 and {max, number}.',
      values: {
        max: 10000
      }
    }));
  }
  return errors;
};
const validateSearchSourceParams = ruleParams => {
  const errors = (0, _lodash.defaultsDeep)({}, _constants.SEARCH_SOURCE_ONLY_EXPRESSION_ERRORS);
  if (!ruleParams.searchConfiguration) {
    errors.searchConfiguration.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredSearchConfiguration', {
      defaultMessage: 'Search source configuration is required.'
    }));
    return errors;
  }
  if (!ruleParams.searchConfiguration.index) {
    errors.searchConfiguration.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredDataViewText', {
      defaultMessage: 'Data view is required.'
    }));
    return errors;
  }
  if (!ruleParams.timeField) {
    errors.timeField.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredDataViewTimeFieldText', {
      defaultMessage: 'Data view should have a time field.'
    }));
    return errors;
  }
  return errors;
};
const validateEsQueryParams = ruleParams => {
  const errors = (0, _lodash.defaultsDeep)({}, _constants.ONLY_ES_QUERY_EXPRESSION_ERRORS);
  if (!ruleParams.index || ruleParams.index.length === 0) {
    errors.index.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredIndexText', {
      defaultMessage: 'Index is required.'
    }));
  }
  if (!ruleParams.timeField) {
    errors.timeField.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredTimeFieldText', {
      defaultMessage: 'Time field is required.'
    }));
  }
  if (!ruleParams.esQuery) {
    errors.esQuery.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredQueryText', {
      defaultMessage: 'Elasticsearch query is required.'
    }));
  } else {
    try {
      const parsedQuery = JSON.parse(ruleParams.esQuery);
      if (!parsedQuery.query) {
        errors.esQuery.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.requiredEsQueryText', {
          defaultMessage: `Query field is required.`
        }));
      }
    } catch (err) {
      errors.esQuery.push(_i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.validation.error.jsonQueryText', {
        defaultMessage: 'Query must be valid JSON.'
      }));
    }
  }
  return errors;
};
const validateExpression = ruleParams => {
  const validationResult = {
    errors: {}
  };
  const commonErrors = validateCommonParams(ruleParams);
  validationResult.errors = commonErrors;

  /**
   * Skip esQuery and index params check if it is search source rule,
   * since it should contain searchConfiguration instead of esQuery and index.
   *
   * It's important to report searchSource rule related errors only into errors.searchConfiguration prop.
   * For example errors.index is a mistake to report searchSource rule related errors. It will lead to issues.
   */
  const isSearchSource = (0, _util.isSearchSourceRule)(ruleParams);
  if (isSearchSource) {
    validationResult.errors = {
      ...validationResult.errors,
      ...validateSearchSourceParams(ruleParams)
    };
    return validationResult;
  }
  const esQueryErrors = validateEsQueryParams(ruleParams);
  validationResult.errors = {
    ...validationResult.errors,
    ...esQueryErrors
  };
  return validationResult;
};
exports.validateExpression = validateExpression;
const hasExpressionValidationErrors = ruleParams => {
  const {
    errors: validationErrors
  } = validateExpression(ruleParams);
  return Object.keys(validationErrors).some(key => validationErrors[key] && validationErrors[key].length);
};
exports.hasExpressionValidationErrors = hasExpressionValidationErrors;